from abc import ABC, abstractmethod
from model_utils import Choices
from xml.sax.saxutils import escape as encode_xml

from django.template.loader import render_to_string

SOUNDS = Choices(
    'Alert',
    'Announcement',
    'Binding',
    'Bump',
    'Busy',
    'CallDisconnect',
    'CallInitiate',
    'CallWaiting',
    'Dial',
    'KeyInput',
    'KeyInputDelete',
    'KeyTone',
    'Nav',
    'NavBack',
    'Notification',
    'OK',
    'Pairing',
    'PresentationConnect',
    'Ringing',
    'SignIn',
    'SpecialInfo',
    'StartListening',
    'TelephoneCall',
    'VideoCall',
    'VolumeAdjust',
    'WakeUp',
)


class AbstractCommand(ABC):
    @property
    @abstractmethod
    def xml_template(self):
        pass

    @abstractmethod
    def __str__(self):
        pass


class RegisterWebhook(AbstractCommand):
    xml_template = '''
        <Command>
           <HttpFeedback>
              <Register>
                 <FeedbackSlot>{slot}</FeedbackSlot>
                 <ServerUrl>{url}</ServerUrl>
                 <Format>JSON</Format>
                 {expressions}
              </Register>
           </HttpFeedback>
        </Command>'''

    def __init__(self, url, expressions, slot=1):
        self.url = url
        if isinstance(expressions, str):
            expressions = [expressions]
        self.expressions = expressions
        self.slot = slot

    def __str__(self):
        expressions = ""
        for number in range(len(self.expressions)):
            expressions += '<Expression item="{number}">{expression}</Expression>'.format(
                number=number + 1,
                expression=self.expressions[number],
            )
        return self.xml_template.format(slot=self.slot, url=self.url, expressions=expressions)


class ChangeLayout(AbstractCommand):
    xml_template = '''
        <Command>
            <UserInterface>
                <Extensions>
                    <Set>
                        <ConfigId>1</ConfigId>
                        <body>{body}</body>
                    </Set>
                </Extensions>
            </UserInterface>
        </Command>'''

    def __init__(self, layout, **context):
        rendered_layout = render_to_string(layout, context=context)
        self.layout = encode_xml(rendered_layout)

    def __str__(self):
        return self.xml_template.format(body=self.layout)


class Dial(AbstractCommand):
    xml_template = '''
        <Command>
            <Dial>
                <Number>{number}</Number>
            </Dial>
        </Command>'''

    def __init__(self, number):
        self.number = number

    def __str__(self):
        return self.xml_template.format(number=self.number)


class SendAlert(AbstractCommand):
    xml_template = '''
    <Command>
        <UserInterface>
            <Message>
                <Alert>
                    <Display>
                        <Title>{title}</Title>
                        <Text>{text}</Text>
                        <Duration>{duration}</Duration>
                    </Display>
                </Alert>
            </Message>
        </UserInterface>
    </Command>'''

    def __init__(self, title, text, duration=0):
        self.title = title
        self.text = text
        self.duration = duration

    def __str__(self):
        return self.xml_template.format(
            title=self.title,
            text=self.text,
            duration=self.duration
        )


class PlaySound(AbstractCommand):
    xml_template = '''
    <Command>
        <Audio>
            <Sound>
                <Play>
                    <Sound>{sound}</Sound>
                </Play>
            </Sound>
        </Audio>
    </Command>'''

    def __init__(self, sound: SOUNDS):
        self.sound = sound

    def __str__(self):
        return self.xml_template.format(sound=self.sound)


class DisplayTextLine(AbstractCommand):
    xml_template = '''
    <Command>
        <UserInterface>
            <Message>
                <TextLine>
                    <Display>
                        <Text>{text}</Text>
                        <X>{x}</X>
                        <Y>{y}</Y>
                        <Duration>{duration}</Duration>
                    </Display>
                </TextLine>
            </Message>
        </UserInterface>
    </Command>'''

    def __init__(self, text: str, x: int = 1, y: int = 1, duration: int = 0):
        self.text = text
        self.x = x
        self.y = y
        self.duration = duration

    def __str__(self):
        return self.xml_template.format(
            text=self.text,
            x=self.x,
            y=self.y,
            duration=self.duration,
        )
