import logging

from datetime import datetime, timedelta

from django.core.management.base import BaseCommand
from django.utils import timezone
from django.db.models import Q

from intranet.vconf.src.rooms.models import UpdateQueue
from intranet.vconf.src.rooms.manager import RoomManager

log = logging.getLogger(__name__)


class Command(BaseCommand):
    def handle(self, *args, **options):
        current_time = datetime.utcnow().replace(tzinfo=timezone.utc)
        updates = get_updates(current_time)
        updated_codecs = update_codecs(updates)
        delete_updates(updated_codecs, earlier_than=(current_time - timedelta(hours=1)))


def get_updates(current_time):
    return list(
        UpdateQueue.objects
        .filter(datetime__lte=current_time)
        .distinct('room__codec_ip')
        .values_list('room__codec_ip')
    )


def update_codecs(codecs):
    updated_codecs = []
    for ip in codecs:
        manager = RoomManager(ip)
        try:
            ips = manager.update_codecs_layout(oneself=True)
            assert len(ips) == 1
            updated_codecs.append(ip)
            log.info('Updated layout for {}'.format(ip))
        except RoomManager.Error:
            log.error('Error when updating layout for {}'.format(ip))

    return updated_codecs


def delete_updates(updated_codecs, earlier_than):
    qs = UpdateQueue.objects.filter(
        Q(room__codec_ip__in=updated_codecs) |
        Q(datetime__lte=earlier_than)
    )
    qs.delete()
