import logging

import yenv

from django.conf import settings
from django.core.management.base import BaseCommand
from django.db.transaction import atomic

from intranet.vconf.src.ext_api.cucm import CUCMClient
from intranet.vconf.src.ext_api.staff import get_all_rooms
from intranet.vconf.src.rooms.models import Room

log = logging.getLogger(__name__)


class Command(BaseCommand):
    def handle(self, *args, **options):
        rooms = get_all_rooms()['result']

        if yenv.type == 'production':
            rooms = filter_production_rooms(rooms)
        else:
            rooms = filter_testing_rooms(rooms)

        rooms = get_extended_room_data_from_cucm(rooms)
        staff_room_id_to_room_map = Room.objects.in_bulk(field_name='room_id')
        new_rooms, rooms_to_update = find_new_and_updated_codec_ips(staff_room_id_to_room_map, list(rooms))

        with atomic():
            Room.objects.bulk_create(new_rooms)
            Room.objects.bulk_update(
                rooms_to_update,
                fields=['codec_ip'],
                batch_size=64,
            )

        log.info('Added %s new codec ips, updated %s', len(new_rooms), len(rooms_to_update))


def filter_production_rooms(rooms: list[dict]):
    non_production_room_ids = settings.PRODUCTION_BLACKLIST_CODEC_ROOM_IDS + settings.TESTING_CODECS_ROOM_IDS
    return [
        room for room in rooms
        if (
            room['id'] not in non_production_room_ids
            and room['floor']['office']['id'] in settings.PRODUCTION_ALLOWED_OFFICE_IDS
        )
    ]


def filter_testing_rooms(rooms: list[dict]):
    return [room for room in rooms if room['id'] in settings.TESTING_CODECS_ROOM_IDS]


def find_new_and_updated_codec_ips(
    staff_room_id_to_room_map: dict[str, Room],
    rooms: list[dict],
) -> tuple[list[Room], list[Room]]:
    new_rooms = []
    rooms_to_update = []
    existing_ips = Room.objects.values_list('codec_ip', flat=True)
    for room in rooms:
        room_id = str(room['id'])
        if room_id in staff_room_id_to_room_map:
            room_obj = staff_room_id_to_room_map[room_id]
            if room_obj.codec_ip != room['ip']:
                room_obj.codec_ip = room['ip']
                rooms_to_update.append(room_obj)
        elif room['ip'] in existing_ips:
            # случай, когда в базе есть комната с другим room_id и таким же ip
            log.error(
                'Room with IPv6 %s already exists. Room %s can not use the same IPv6',
                room['ip'],
                room['name']['exchange'],
            )
        else:
            new_rooms.append(
                Room(
                    codec_ip=room['ip'],
                    room_id=room_id,
                    name=room['name']['display'],
                    email=f'{room["name"]["exchange"]}@yandex-team.ru',
                    office_id=str(room['floor']['office']['id']),
                    timezone=room['floor']['office']['timezone'],
                ),
            )

    return new_rooms, rooms_to_update


def get_extended_room_data_from_cucm(rooms: list[dict]):
    """
    Обогащает данные о комнатах ip-адресом или удаляет комнату из списка,
    если в CUCM ip-адреса нет
    """
    room_numbers = [
        room['equipment']['video_conferencing'] for room in rooms
        if room['equipment']['video_conferencing'] and room['equipment']['video_conferencing'].isdigit()
    ]
    client = CUCMClient(service_type='RIS')
    codec_number_to_ipv6_and_model_map = client.get_codec_number_to_ipv6_and_model_map(room_numbers)

    extended_rooms = []
    used_ips = set()

    for room in rooms:
        number = room['equipment']['video_conferencing']

        if number not in codec_number_to_ipv6_and_model_map:
            continue

        codec_model_supported = (
            number
            and codec_number_to_ipv6_and_model_map[number]['model'] not in settings.UNSUPPORTED_CODEC_MODEL_IDS
        )

        if codec_model_supported:
            room['ip'] = codec_number_to_ipv6_and_model_map[number]['ip']
            if room['ip'] in used_ips:
                log.error(
                    'Two rooms has the same IPv6 %s, one of them is %s',
                    room['ip'],
                    room['name']['exchange'],
                )
            else:
                used_ips.add(room['ip'])
                extended_rooms.append(room)

    return extended_rooms
