from datetime import datetime, timedelta
import logging

from django.core.management.base import BaseCommand

from intranet.vconf.src.rooms.models import Room, Event
from intranet.vconf.src.rooms.manager import RoomManager
from intranet.vconf.src.ext_api.calendar import get_events_by_time


log = logging.getLogger(__name__)


class Command(BaseCommand):
    def handle(self, *args, **options):
        rooms = get_rooms_to_update()
        update_codecs_layout(rooms)


def get_rooms_to_update():
    five_minutes_later = datetime.utcnow() + timedelta(0, 300)
    office_ids = get_office_ids()
    events = get_events_by_time(office_ids, five_minutes_later)
    events_info, events_rooms = parse_events(events)

    new_events = create_new_events(events_info)
    update_rooms_event(new_events, events_rooms)

    updated_events = update_events(events_info, events_rooms)
    update_rooms_event(updated_events, events_rooms)

    rooms_with_ended_events = delete_ended_events(events_info)

    values = ['codec_ip']
    rooms = RoomManager.get_rooms_for_events(
        event_ids=new_events + updated_events,
        values=values
    )
    rooms += RoomManager.get_rooms_by_codec_ips(
        codec_ips=rooms_with_ended_events,
        values=values
    )
    return rooms


def parse_events(events):
    events_info = {}
    events_rooms = {}
    for event_id, event in events.items():
        try:
            events_info[event_id] = {
                'event_id': event_id,
                'name': event['name'],
                'start_time': event['start_time'],
                'end_time': event['end_time'],
                'organizer': event['organizer'],
            }
            events_rooms[event_id] = event['rooms']
        except KeyError:
            log.exception('Incorrect data from Calendar API')
    return events_info, events_rooms


def get_office_ids():
    return list(Room.objects.all().values_list('office_id', flat=True).distinct())


def delete_ended_events(events_info):
    qs = Event.objects.all().exclude(event_id__in=events_info.keys())
    ended_events = qs.values_list('event_id', flat=True)
    rooms = list(
        Room.objects
        .filter(event__event_id__in=ended_events)
        .values_list('codec_ip', flat=True)
    )
    qs.delete()
    return rooms


def create_new_events(events_info):
    existing_events = (
        Event.objects
        .filter(event_id__in=events_info.keys())
        .values_list('event_id', flat=True)
    )
    new_events = list(set(events_info.keys()) - set(existing_events))
    (
        Event.objects
        .bulk_create([
            Event(
                **events_info[event_id]
            ) for event_id in new_events
        ])
    )
    return new_events


def update_events(events_info, events_rooms):
    all_emails = set(Room.objects.all().values_list('email', flat=True))
    qs = Event.objects.filter(event_id__in=events_info.keys())
    for event_id in events_info.keys():
        emails_in_database = set(
            Room.objects
            .filter(event__event_id=event_id)
            .values_list('email', flat=True)
        )
        available_emails = all_emails & set(events_rooms[event_id])
        if emails_in_database == available_emails:
            qs = qs.exclude(**events_info[event_id])

    events_to_update = qs.values_list('event_id', flat=True)
    for event_id in events_to_update:
        Event.objects.filter(event_id=event_id).update(**events_info[event_id])

    return list(events_to_update)


def update_rooms_event(event_ids, events_rooms):
    events = Event.objects.filter(event_id__in=event_ids).values_list('id', 'event_id')
    for row_id, event_id in events:
        Room.objects.filter(email__in=events_rooms[event_id]).update(event=row_id)


def update_codecs_layout(rooms):
    codecs = set([room['codec_ip'] for room in rooms if room['codec_ip'] is not None])
    while len(codecs) != 0:
        codec = next(iter(codecs))
        try:
            manager = RoomManager(codec)
            updated_codecs = manager.update_codecs_layout()
        except RoomManager.Error:
            codecs.remove(codec)
            log.error('Error when updating layout for codec with ip "%s"', codec)
        else:
            log.info('Updated layout for {}'.format(updated_codecs))
            codecs_size = len(codecs)
            codecs -= set(updated_codecs)
            assert codecs_size != len(codecs)
