import logging

from django.core.management.base import BaseCommand

from intranet.vconf.src.ext_api.staff import get_all_rooms
from intranet.vconf.src.ext_api.calendar import get_events_for_offices
from intranet.vconf.src.rooms.models import Room

log = logging.getLogger(__name__)


class Command(BaseCommand):
    def handle(self, *args, **options):
        rooms = get_all_rooms()['result']
        rooms = fix_staff_api_response(rooms)
        rooms = get_rooms_with_valid_office_id(rooms)
        rooms = get_new_rooms(rooms)
        create_rooms(rooms)


def fix_staff_api_response(rooms):
    result = [room for room in rooms if room['name']['exchange'] != '']
    for room in result:
        room['id'] = str(room['id'])
    return result


def get_rooms_with_valid_office_id(rooms):
    office_ids = [room['floor']['office']['id'] for room in rooms]
    invalid_ids = get_invalid_office_ids(office_ids)
    return [room for room in rooms if room['floor']['office']['id'] not in invalid_ids]


def get_invalid_office_ids(office_ids):
    office_ids = list(set(office_ids))
    result = []
    for office in office_ids:
        try:
            get_events_for_offices(office)
        except KeyError:
            result.append(office)
    return result


def get_new_rooms(rooms):
    room_ids = [room['id'] for room in rooms]
    existing_rooms = Room.objects.filter(room_id__in=room_ids).values_list('room_id', flat=True)
    new_room_ids = set(room_ids) - set(existing_rooms)
    return [room for room in rooms if room['id'] in new_room_ids]


def create_rooms(rooms):
    log.info([room for room in rooms if room['name']['exchange'] == ''])
    Room.objects.bulk_create([
        Room(
            name=room['name']['display'],
            room_id=room['id'],
            email='{}@yandex-team.ru'.format(room['name']['exchange']),
            office_id=room['floor']['office']['id'],
            codec_ip=None,
            timezone=room['floor']['office']['timezone'],
            language=room['floor']['office']['city']['country']['code']
        )
        for room in rooms
    ])
