from base64 import b64encode
import logging
import pytz
import requests
from typing import List, Union

from django.contrib.auth import get_user_model
from django.conf import settings
from django.core.exceptions import ObjectDoesNotExist
from django.utils import translation

from intranet.vconf.src.ext_api.calendar import get_event_info, EventDoesNotExist
from intranet.vconf.src.ext_api.staff import get_room_info
from intranet.vconf.src.lib.requests import Session
from intranet.vconf.src.lib.utils import round_duration
from intranet.vconf.src.call.constants import CALL_STATES
from intranet.vconf.src.call.event import find_zoom_link
from intranet.vconf.src.call.hydrator import EventsHydrator, ParticipantsHydrator
from intranet.vconf.src.call.manager import CallManager
from intranet.vconf.src.call.models import PARTICIPANT_TYPES, CALL_METHODS
from intranet.vconf.src.call.participant_ctl import ParticipantCtl, ArbitraryParticipant

from intranet.vconf.src.rooms.commands import (
    AbstractCommand,
    ChangeLayout,
    Dial,
    PlaySound,
    RegisterWebhook,
    SendAlert,
    SOUNDS,
)
from intranet.vconf.src.rooms.models import Room, Event
from intranet.vconf.src.rooms.utils import room_in_beta_test_list


log = logging.getLogger(__name__)
User = get_user_model()


class RoomManager:
    class Error(Exception):
        def __init__(self, message):
            log.exception(message)
            super(RoomManager.Error, self).__init__(message)

    def __init__(self, codec_ip):
        try:
            self.obj = self.find_room_by_ip(codec_ip)
            self.codec_ip = self.obj.codec_ip
        except Room.DoesNotExist:
            raise RoomManager.Error(f'Not found codec with ip "{codec_ip}"')

    @staticmethod
    def find_room_by_ip(codec_ip: str):
        try:
            # сначала ищем кодек по последним четырем числам IPv6 адреса (eui64)
            # если такие eui64 совпадают у каких-то кодеков, ищем по полному IPv6 адресу
            codec_eui64 = ':'.join(codec_ip.split(':')[-4:])
            return Room.objects.get(codec_ip__contains=codec_eui64)
        except Room.MultipleObjectsReturned:
            return Room.objects.get(codec_ip=f'[{codec_ip}]')

    def send_command(self, command: AbstractCommand):
        session = Session()
        try:
            credentials = get_codec_credentials()
            response = session.post(
                'https://{host}/putxml'.format(host=self.codec_ip),
                data=str(command).encode('utf-8'),
                timeout=3,
                headers={'Authorization': 'Basic {}'.format(credentials)},
                verify=False
            )
            response.raise_for_status()
        except requests.exceptions.Timeout:
            raise self.Error('Host {} timed out'.format(self.codec_ip))
        except requests.exceptions.HTTPError:
            if response.status_code == 500:
                raise self.Error('Check credentials for {}'.format(self.codec_ip))
            if response.status_code == 400:
                raise self.Error('XML syntax error {}'.format(self.codec_ip))
            raise self.Error('Codec {} is not available'.format(self.codec_ip))
        except requests.exceptions.ConnectionError:
            raise self.Error('Bad codec ip {}'.format(self.codec_ip))

        if 'error' in response.content.decode('utf-8').lower():
            raise self.Error('Bad request body: {}'.format(response.request.body))

    def register_webhook(self, url: str, expressions: Union[str, List[str]]):
        command = RegisterWebhook(url=url, expressions=expressions)
        self.send_command(command)

    def update_layout(self, layout='inactive.xml', language='ru', **context):
        with translation.override(language):
            command = ChangeLayout(layout, **context)
        self.send_command(command)

    def dial(self, number: str):
        """
        Отправляет на кодек команду позвонить на номер number
        """
        command = Dial(number)
        self.send_command(command)

    def notify_meeting_ends(self):
        sound_command = PlaySound(sound=SOUNDS.Announcement)
        alert_command = SendAlert(
            title='The meeting will end soon',
            text='Sum up the results and get ready to leave the meeting room',
            duration=60,
        )
        self.send_command(sound_command)
        self.send_command(alert_command)

    def notify_call_is_being_created(self):
        self.update_layout(
            layout='active.xml',
            event={
                'name': self.obj.event.name,
                'start_time': self.obj.event.start_time,
                'end_time': self.obj.event.end_time,
                'organizer': self.obj.event.organizer,
                'call_id': self.obj.event.call_id,
            },
        )
        command = PlaySound(sound=SOUNDS.Announcement)
        self.send_command(command)

    def notify_call_creation_failed(self):
        command = SendAlert(
            title='Failed to create a call',
            text='Please try again',
        )
        self.send_command(command)

    def create_or_join_call(self, oneself: bool = False):
        """
        Создает новый звонок, если для события, в котором участвует комната,
        нет активного звонка
        Если активный звонок есть, подсоединяет к нему комнату
        :param oneself: если True, то в новый звонок подключается только
         переговорка, которая звонила, иначе созваниваются все переговорки из встречи
        """
        robot = User.objects.get(username=settings.VCONF_ROBOT_LOGIN)
        event = self.get_event()
        if event is None:
            self.update_codecs_layout(oneself=True)
            raise self.Error('There is no event for {}'.format(self.codec_ip))

        try:
            event_info = get_event_info(event_id=event.event_id)
        except EventDoesNotExist:
            log.error('There is no event id=%s in Calendar API', event.id)
            event.delete()
            self.update_layout(layout='inactive.xml')
            return

        zoom_link = find_zoom_link(event_info['description'])
        zoom_enabled = (
            zoom_link
            and settings.CODEC_ZOOM_CALLS_ENABLED
            and room_in_beta_test_list('CODEC_TEST_ZOOM_CALL_ROOM_IDS', self.obj.room_id)
        )

        call = CallManager.get_active_by_event_id(
            event_id=int(event.event_id),
            for_user=robot,
        )

        if call is None:
            if zoom_enabled:
                self.create_zoom_call(event, zoom_link, oneself=oneself)
            else:
                self.create_cms_call(event, zoom_link, oneself=oneself)
        else:
            if event.call_id is None:
                event.call_id = call.obj.conf_cms_id
                event.save()

            if zoom_enabled:
                self.join_zoom_call(zoom_link)
            else:
                self.join_cms_call(call)

    @staticmethod
    def get_zoom_participant(zoom_link: str) -> ArbitraryParticipant:
        return ArbitraryParticipant.from_raw(
            raw_data={
                'id': zoom_link,
                'type': PARTICIPANT_TYPES.arbitrary,
                'method': CALL_METHODS.zoom,
            },
            safe=True,
        )

    def create_zoom_call(self, event: Event, zoom_link: str, oneself: bool = False):
        """
        Отправляет команды подключения к зуму
        Звонок в CMS при это не создается
        :param oneself: если True, то в новый звонок подключается только
         переговорка, которая звонила, иначе созваниваются все переговорки из встречи
        """
        zoom_participant = self.get_zoom_participant(zoom_link)

        number = zoom_participant.data['id']

        if oneself:
            self.dial(number)
            log.info(
                'Created Zoom call via Cisco button: event_id = %s, codec_ip = %s started a call',
                event.event_id,
                self.codec_ip,
            )
            return

        rooms = Room.objects.filter(event_id=event.id)
        rooms_cnt = 0

        for room in rooms:
            try:
                mngr = RoomManager(room.codec_ip)
                mngr.dial(number)
                rooms_cnt += 1
            except self.Error:
                log.exception('Can not send Dial command to codec with ip "%s"', room.codec_ip)

        log.info(
            'Created Zoom call via Cisco button: event_id = %s, %s of %s rooms added to call',
            event.event_id,
            rooms_cnt,
            len(rooms),
        )

    def create_cms_call(self, event: Event, zoom_link: str = None, oneself: bool = False):
        """
        Создает VConf-звонок, в который каждая комната подключается как отдельный участник
        Если передается ссылка на zoom, то zoom добавляется как участник звонка
        :param oneself: если True, то в новый звонок подключается только
         переговорка, которая звонила, иначе созваниваются все переговорки из встречи
        """
        try:
            if oneself:
                participants_for_call = [self.as_participant()]
            else:
                participants_for_call = self.get_participants_by_event(event)
        except EventDoesNotExist:
            log.error('There is no event id=%s in Calendar API', event.id)
            event.delete()
            self.update_layout(layout='inactive.xml')
            return

        if zoom_link:
            participants_for_call.append(self.get_zoom_participant(zoom_link))

        self.create_call(event, participants_for_call)

        log.info('Created CMS call via Cisco button: event_id = %s', event.event_id)

    def create_call(self, event: Event, participants: List[ParticipantCtl]):
        """
        Создает звонок для события event с участниками из списка participants
        """
        event_duration = (event.end_time - event.start_time).seconds // 60
        conf_data = {
            'name': event.name,
            'author_login': settings.VCONF_ROBOT_LOGIN,
            'duration': round_duration(event_duration),
            'stream': False,
            'record': False,
            'participants': participants,
            'event_id': event.event_id,
        }

        try:
            author = User.objects.get(username=settings.VCONF_ROBOT_LOGIN)
            call = CallManager.create(author=author, data=conf_data)
        except CallManager.Error as e:
            raise self.Error(str(e))

        self.set_call(call.obj.conf_cms_id)
        return call.obj.conf_cms_id

    def join_zoom_call(self, zoom_link: str):
        """
        Подключает комнату к zoom-звонку как отдельного участника
        (отправляет на кодек команду позвонить по нужному номеру)
        """
        zoom_number = ArbitraryParticipant.normalize_zoom_url(zoom_link)
        self.dial(zoom_number)
        log.info('Joined Zoom call via Cisco button: Zoom number = %s', zoom_number)

    def join_cms_call(self, call: CallManager):
        """
        Подключает комнату к звонку в VConf как отдельного участника
        """
        call.add_participant(self.as_participant())
        log.info('Joined CMS call via Cisco button: conf_cms_id = %s', call.obj.conf_cms_id)

    def stop_call(self, conf_cms_id: str):
        user = User.objects.get(username=settings.VCONF_ROBOT_LOGIN)
        try:
            mngr = CallManager.get_by_id(
                for_user=user,
                conf_cms_id=conf_cms_id,
            )
        except ObjectDoesNotExist:
            log.error('Call %s does not exist', conf_cms_id)
            return

        if mngr.obj.state == CALL_STATES.active:
            try:
                mngr.stop()
            except CallManager.Error as e:
                raise self.Error(str(e))
        else:
            log.error('Call %s is not active', conf_cms_id)

        self.unset_call()

    def set_call(self, call_id: str):
        event = self.get_event()
        if event is not None:
            event.call_id = call_id
            event.save()

    def unset_call(self):
        event = self.get_event()
        if event is not None:
            event.call_id = None
            event.save()

    def update_codecs_layout(self, oneself=False):
        event = self.get_event()
        if event is None:
            self.update_layout(layout='inactive.xml')
            return [self.codec_ip]

        rooms = self.get_rooms_for_event(
            event_id=event.event_id,
            values=[
                'codec_ip',
                'timezone',
                'name',
                'language',
                'event__name',
                'event__start_time',
                'event__end_time',
                'event__organizer',
                'event__id',
                'event__call_id',
            ]
        )

        event_room_names = [room['name'] for room in rooms]

        if oneself:
            rooms = list(filter(lambda room: room['codec_ip'] == self.codec_ip, rooms))

        layout = 'event.xml' if event.call_id is None else 'active.xml'

        for room in rooms:
            change_timezone_to_local(room)
            manager = RoomManager(room['codec_ip'])
            try:
                manager.update_layout(
                    layout=layout,
                    language=room['language'],
                    event={
                        'name': room['event__name'],
                        'start_time': room['event__start_time'],
                        'end_time': room['event__end_time'],
                        'organizer': room['event__organizer'],
                        'rooms': event_room_names,
                        'call_id': room['event__call_id'],
                    },
                )
            except self.Error:
                if oneself:
                    raise self.Error('Error when updating layout for {}'.format(room['codec_ip']))

        return [room['codec_ip'] for room in rooms]

    @staticmethod
    def get_rooms_by_codec_ips(codec_ips: List[str], values: List[str]) -> List[dict]:
        return list(Room.objects.filter(codec_ip__in=codec_ips).values(*values))

    @staticmethod
    def get_rooms_for_event(event_id: str, values: List[str]) -> List[dict]:
        return RoomManager.get_rooms_for_events([event_id], values)

    @staticmethod
    def get_rooms_for_events(event_ids: List[str], values: List[str]) -> List[dict]:
        return list(Room.objects.filter(event__event_id__in=event_ids).values(*values))

    @staticmethod
    def get_participants_by_event(event: Event) -> List[ParticipantCtl]:
        event_info = get_event_info(event_id=event.event_id)
        user = User.objects.get(username=settings.VCONF_ROBOT_LOGIN)
        hydrator = EventsHydrator(user)
        hydrator.add_to_fetch([event_info])
        participants = hydrator.hydrate(event_info).rooms

        return participants

    def as_participant(self) -> ParticipantCtl:
        room = Room.objects.get(codec_ip=self.codec_ip)
        room_info = get_room_info([room.room_id])['result'][0]
        pt = {
            'type': PARTICIPANT_TYPES.room,
            'id': room_info['id'],
            'method': CALL_METHODS.cisco,
            'action': ['cisco'],
            'number': room_info['phone'],
        }
        hydrator = ParticipantsHydrator('en')
        hydrator.add_to_fetch([pt])
        return hydrator.hydrate(pt)

    def get_event(self) -> Event:
        try:
            event = Event.objects.get(room__codec_ip=self.codec_ip)
        except Event.DoesNotExist:
            event = None

        return event


def change_timezone_to_local(room: dict):
    tz = pytz.timezone(room['timezone'])
    try:
        room['event__start_time'] = tz.normalize(room['event__start_time'])
        room['event__start_time'] = room['event__start_time'].replace(tzinfo=None)
        room['event__end_time'] = tz.normalize(room['event__end_time'])
        room['event__end_time'] = room['event__end_time'].replace(tzinfo=None)
    except KeyError:
        pass


def get_codec_credentials() -> str:
    username = bytes('ld\\' + settings.VCONF_ROBOT_LOGIN, encoding='utf-8')
    pwd = bytes(settings.VCONF_ROBOT_PASSWORD, encoding='utf-8')
    return b64encode(username + b':' + pwd).decode('ascii')
