import json
import logging

from django.http import HttpResponse

from intranet.vconf.src.rooms.exceptions import (
    IncorrectCodecEventBody,
    IncorrectIPv6Address,
    UnknownUIExtension,
)
from intranet.vconf.src.rooms.manager import RoomManager
from intranet.vconf.src.rooms.utils import is_ipv6_address, get_request_ipv6

log = logging.getLogger(__name__)


def webhook(request):
    """
    Вебхук для обращения с планшетов Cisco
    Даже если происходит ошибка, нужно возвращать 200, иначе у кодека оторвется адрес вебхуки
    """

    try:
        body = json.loads(request.body.decode('utf-8'))
    except json.JSONDecodeError:
        log.error('Error decoding JSON')
        return HttpResponse(status=200)
    except UnicodeDecodeError:
        log.error('Error decoding Unicode')
        return HttpResponse(status=200)

    try:
        codec_event = parse_codec_event(body)
        codec_ip = get_request_ipv6(request) or get_codec_ip(body)
    except IncorrectCodecEventBody as err:
        log.error('Bad codec event body: %s', err.body)
        return HttpResponse(status=200)
    except IncorrectIPv6Address as err:
        log.error('Bad codec event IPv6: %s', err.address)
        return HttpResponse(status=200)
    except UnknownUIExtension as err:
        log.error('Unknown UI extension: %s', err.extension)
        return HttpResponse(status=200)

    try:
        manager = RoomManager(codec_ip)
    except RoomManager.Error:
        return HttpResponse(status=200)

    if codec_event['action'] == 'clicked':
        if codec_event['id'] in ['start_call_for_all', 'start_call_for_me', 'join_call']:
            manager.notify_call_is_being_created()
            try:
                oneself = (codec_event['id'] == 'start_call_for_me')
                manager.create_or_join_call(oneself=oneself)
            except RoomManager.Error:
                manager.notify_call_creation_failed()
                log.error('Error when creating call for %s', codec_ip)
        elif codec_event['id'].startswith('stop_call'):
            conf_cms_id = codec_event['id'].split('_')[2]
            try:
                manager.stop_call(conf_cms_id)
            except RoomManager.Error:
                log.error('Error when stopping call %s for %s', conf_cms_id, codec_ip)

        manager.update_codecs_layout()

    return HttpResponse(status=200)


def get_codec_ip(body: dict) -> str:
    try:
        address = body['Event']['Identification']['IPAddressV6']['Value']
    except KeyError:
        raise IncorrectCodecEventBody(body)

    if not is_ipv6_address(address):
        raise IncorrectIPv6Address(address)

    return address


def parse_codec_event(body: dict) -> dict:

    try:
        ui = body['Event']['UserInterface']['Extensions']
    except KeyError:
        raise IncorrectCodecEventBody(body)

    if 'Panel' in ui:
        return {
            'id': ui['Panel']['Clicked']['PanelId']['Value'],
            'action': 'clicked',
        }
    elif 'Widget' in ui:
        return {
            'id': ui['Widget']['Action']['WidgetId']['Value'],
            'action': ui['Widget']['Action']['Type']['Value'],
        }

    raise UnknownUIExtension(ui)
