import logging

from django.contrib.auth.models import AbstractUser
from django.utils.functional import cached_property

from intranet.vconf.src.ext_api.staff import get_person_info, is_ip_internal


logger = logging.getLogger(__name__)


class User(AbstractUser):

    def __init__(self, *args, **kwargs):
        self._extra = kwargs.pop('extra', {})
        super().__init__(*args, **kwargs)

    @property
    def extra(self):
        """
        Поля, которые берутся из паспорта или из staff-api.
        Нужно, чтобы оставить обратную совместимость со старым User.
        :return:
        """
        return self._extra

    @property
    def login(self):
        return self.username

    @cached_property
    def person_info(self):
        return get_person_info([self.username])['result'][0]

    @property
    def affiliation(self):
        if 'affiliation' in self.extra:
            return self.extra['affiliation']
        return self.person_info['official']['affiliation']

    @cached_property
    def tz(self):
        # TODO: брать из паспорта
        if 'tz' in self.extra:
            return self.extra['tz']
        return self.person_info['environment']['timezone']

    @property
    def uid(self):
        return self.extra.get('uid')

    @property
    def lang(self):
        return self.extra.get('lang')

    @property
    def is_external(self):
        if 'is_external' in self.extra:
            return self.extra['is_external']
        result = self.affiliation not in ('yandex', 'yamoney')
        # Сейчас фронт завязан на флаг is_external, чтобы показывать вкладку трансляций.
        # Надо отдавать на фронт флаг can_see_stream, а is_external отдавать по честному.
        # Будем править в VCONF-531
        if self.is_stream_viewer:
            result = False
        return result

    @property
    def can_see_streams(self):
        return not self.is_external

    @property
    def is_ip_external(self):
        if 'is_ip_external' in self.extra:
            return self.extra['is_ip_external']
        try:
            return not is_ip_internal(self.extra.get('ip'))
        except Exception:
            logger.exception('Failed to get data from auth-checker.')
            return True

    @property
    def call_secret(self):
        return self.extra.get('secret')

    @property
    def user_ticket(self):
        return self.extra.get('user_ticket')

    @property
    def is_admin(self):
        return self.has_perm('permissions.admin_perm')

    @property
    def is_stream_viewer(self):
        return self.has_perm('permissions.stream_viewer_perm')

    class Meta:
        db_table = 'auth_user'
