from fastapi import status
from fastapi_utils.cbv import cbv
from fastapi_utils.inferring_router import InferringRouter

from watcher.api.routes.base import BaseRoute
from watcher.api.schemas.abc_migration import AbcMigrationListSchema
from watcher.api.schemas.base import CursorPaginationResponse
from watcher.logic.exceptions import PermissionDenied
from watcher.logic.permissions import is_superuser
from watcher.logic.service import (
    is_responsible_in_service
)
from watcher.db import AbcMigration
from watcher.tasks.abc_migration import (
    create_migrations,
    finalize_migrations,
)


router = InferringRouter()


@cbv(router)
class ServiceRoute(BaseRoute):
    model = AbcMigration

    @router.get('/{migration_id}')
    def retrieve(self, migration_id: int) -> AbcMigrationListSchema:
        abc_migration = self.get_object(object_id=migration_id)
        return abc_migration

    @router.get('/')
    def list(self) -> CursorPaginationResponse[AbcMigrationListSchema]:
        pagination_response = self.list_objects()
        return pagination_response

    @router.post('/{service_id}/prepare', status_code=status.HTTP_204_NO_CONTENT)
    def prepare(self, service_id: int):
        self._check_migrate_permissions(service_id=service_id)
        create_migrations(session=self.session, service_id=service_id, author_id=self.current_user.id)

    @router.post('/{service_id}/finalize', status_code=status.HTTP_204_NO_CONTENT)
    def finalize(self, service_id: int):
        self._check_migrate_permissions(service_id=service_id)
        finalize_migrations(session=self.session, service_id=service_id)

    def _check_migrate_permissions(self, service_id: int):
        if not is_superuser(staff=self.current_user):
            if not is_responsible_in_service(
                db=self.session, staff_id=self.current_user.id,
                service_id=service_id
            ):
                raise PermissionDenied(message={
                    'ru': 'Нет прав на миграцию расписания сервиса',
                    'en': 'No permission to migrate service schedules',
                })
