import logging
from typing import Optional

from fastapi import Depends, Request
from sqlalchemy.orm import Session
from sqlalchemy.orm.query import Query

from watcher.db.base import get_db, BaseModel
from watcher.db import Staff, Schedule
from watcher.api.schemas.base import (
    CursorPaginationResponse,
    BaseSchema,
)
from watcher.crud.base import (
    get_object_by_model_or_404,
    list_objects_by_model,
    filter_objects,
)
from watcher.logic.filter import get_filter_params
from watcher.logic.fields import get_fields_params
from watcher.logic.pagination import (
    get_cursor_param,
    get_page_size_param,
    CursorPagination,
)
from watcher.logic.exceptions import PermissionDenied
from watcher.logic.order import (
    get_order_by_params,
)
from watcher.crud.staff import get_staff_by_uid
from watcher.crud.base import commit_object
from watcher.logic.permissions import (
    has_strictly_or_limited_or_full_role,
    has_full_access,
    is_superuser,
    schedules_responsible_in,
)
logger = logging.getLogger(__name__)


class BaseRoute:
    session: Session = Depends(get_db)
    model = None
    filter_params: Optional[dict] = Depends(get_filter_params)
    cursor: Optional[str] = Depends(get_cursor_param)
    page_size: Optional[int] = Depends(get_page_size_param)
    ordering: Optional[tuple] = Depends(get_order_by_params)
    default_ordering = '-id',
    joined_load = tuple()
    joined_load_list = tuple()
    fields_params: Optional[dict] = Depends(get_fields_params)

    def __init__(self, request: Request):
        self.request = request

        # пришли только с service ticket'ом
        service_ticket_only = request.scope['service_ticket_only']

        if service_ticket_only:
            if self.request.method != 'GET':
                raise PermissionDenied(message={
                    'ru': 'С X-Ya-Service-Ticket доступны только GET запросы',
                    'en': 'With X-Ya-Service-Ticket only GET requests are available',
                })
        else:
            if not is_superuser(staff=self.current_user):
                self._check_granular_permissions()
                if self.request.method in ('DELETE', 'PATCH', 'PUT'):
                    self._check_change_permissions()

    def _check_change_permissions(self):
        pass

    def _check_granular_permissions(self):
        """
        Данный метод проверяет базовые права - гранулярный доступ.
        у сильно ограниченной роли есть доступ на GET
        у полной роли есть доступ на все остальные методы
        """
        if (
            self.request.method == 'GET' and
            not has_strictly_or_limited_or_full_role(db=self.session, user_id=self.current_user.user_id)
        ):
            raise PermissionDenied(message={
                'ru': 'У пользователя отсутствует сильно ограниченная роль',
                'en': 'User does not have strictly limited role',
            })

        if (
            self.request.method in ('DELETE', 'POST', 'PATCH', 'PUT') and
            not has_full_access(db=self.session, user_id=self.current_user.user_id)
        ):
            raise PermissionDenied(message={
                'ru': 'У пользователя отсутствует расширенная роль',
                'en': 'User does not have advanced role',
            })

    @property
    def current_user(self) -> Staff:
        current_user = getattr(self, '_current_user', None)
        if not current_user:
            current_user = get_staff_by_uid(
                db=self.session,
                uid=self.request.user.uid,
            )
            setattr(self, '_current_user', current_user)
        return current_user

    def get_object(self, object_id: int, extra_joined: Optional[tuple] = None) -> BaseModel:
        if getattr(self, '_object_id', None) != object_id:
            joined_load = self.joined_load
            if extra_joined:
                joined_load = joined_load + extra_joined
            setattr(self, '_object', get_object_by_model_or_404(
                db=self.session,
                model=self.model,
                object_id=object_id,
                joined_load=joined_load,
            ))
            setattr(self, '_object_id', object_id)
        return getattr(self, '_object')

    def filter_objects(self, query: Query) -> Query:
        filter_params = self.filter_params
        return filter_objects(query, filter_params)

    def _get_ordering(self) -> tuple:
        return self.ordering or self.default_ordering

    def paginate_objects(self, query: Query) -> CursorPaginationResponse:
        paginator = CursorPagination(
            model=self.model,
            cursor=self.cursor,
            query=query,
            page_size=self.page_size,
            request=self.request,
            ordering=self._get_ordering(),
        )
        next_url, previous_url, items = paginator.paginate()
        return CursorPaginationResponse(
            next=next_url,
            prev=previous_url,
            result=items,
        )

    def list_objects_by_model(self) -> Query:
        return list_objects_by_model(
            db=self.session, model=self.model,
            joined_load=self.joined_load_list,
        )

    def list_objects(self) -> CursorPaginationResponse:
        query = self.list_objects_by_model()
        filtered_query = self.filter_objects(query=query)
        paginated_query = self.paginate_objects(query=filtered_query)
        return paginated_query

    def get_changed_fields(self, obj: BaseModel, schema: BaseSchema) -> list:
        update_data = schema.dict(exclude_unset=True)
        changed_fields = []
        for field, value in update_data.items():
            current_value = getattr(obj, field)
            if current_value != value:
                changed_fields.append(field)
        return changed_fields

    def patch_object(self, obj: BaseModel, schema: BaseSchema) -> BaseModel:
        update_data = schema.dict(exclude_unset=True)
        has_changes = False
        for field, value in update_data.items():
            current_value = getattr(obj, field)
            if current_value != value:
                has_changes = True
                setattr(obj, field, value)

        if has_changes:
            obj = commit_object(db=self.session, obj=obj)
        else:
            logger.info(f'Patch request was made without any changes: {obj}, author: {self.current_user.uid}')
        return obj

    def add_is_responsible(self, items: list[Schedule], field_param: str = 'is_responsible') -> None:
        if field_param in self.fields_params:
            schedules_responsible = schedules_responsible_in(
                db=self.session, staff=self.current_user,
                schedules=items,
            )

            for schedule in items:
                setattr(
                    schedule, 'is_responsible',
                    schedule.id in schedules_responsible
                )
