import datetime
import logging
from typing import Optional, Union

from fastapi import status
from fastapi_utils.cbv import cbv
from fastapi_utils.inferring_router import InferringRouter
from sqlalchemy import or_, func
from sqlalchemy.orm import Query

from watcher.api.routes.base import BaseRoute
from watcher.api.schemas.base import CursorPaginationResponse
from watcher.api.schemas.gap_settings import (
    ManualGapSettingsListSchema,
    ManualGapSettingsCreateSchema,
    ManualGapSettingsPatchSchema,
)

from watcher import enums
from watcher.db import (
    ManualGapSettings,
    ManualGapSettingsServices,
    ManualGapSettingsSchedules,
    Schedule,
    Service,
    Member,
)
from watcher.crud.manual_gap import (
    create_gap_settings,
    patch_gap_settings,
)
from watcher.logic.exceptions import (
    PermissionDenied,
    RecordNotFound,
    ScheduleServiceNotInTheList,
    ManualGapStartInPast,
    InvalidManualGapLength,
    BadRequest,
)
from watcher.logic.timezone import today, localize
from watcher.logic.permissions import is_superuser
from watcher.tasks import (
    update_manual_gaps,
)


logger = logging.getLogger(__name__)
router = InferringRouter()


@cbv(router)
class ManualGapSettingsRoute(BaseRoute):
    model = ManualGapSettings
    joined_load = ('services', 'schedules', 'gaps')
    joined_load_list = ('services', 'schedules', 'gaps')

    @router.get('/{gap_settings_id}')
    def retrieve(self, gap_settings_id: int) -> ManualGapSettingsListSchema:
        return self.get_object(gap_settings_id)

    @router.get('/')
    def list(self) -> CursorPaginationResponse[ManualGapSettingsListSchema]:
        return self.list_objects()

    def filter_objects(self, query: Query) -> Query:
        schedule_id = self.filter_params.pop('schedule_id', None)
        schedule_id_in = self.filter_params.pop('schedule_id__in', None)
        service_id = self.filter_params.pop('service_id', None)
        service_id_in = self.filter_params.pop('service_id__in', None)

        service_ids = [service_id] if service_id else service_id_in
        service_ids = list(map(int, service_ids)) if service_ids else []
        schedule_ids = [schedule_id] if schedule_id else schedule_id_in
        schedule_ids = list(map(int, schedule_ids)) if schedule_ids else []

        if service_ids and schedule_ids:
            raise BadRequest(
                message={
                    'ru': 'Фильтрация одновременно и по service_id и по schedule_id недоступна',
                    'en': 'Filtering by both service_id and schedule_id is not available',
                }
            )
        if service_ids or schedule_ids:
            if not service_ids:
                schedules = self.session.query(Schedule).filter(Schedule.id.in_(schedule_ids))
                service_ids = [schedule.service_id for schedule in schedules]

            query = query.join(
                Member, Member.staff_id == ManualGapSettings.staff_id
            ).filter(
                Member.service_id.in_(service_ids),
                Member.state == enums.MemberState.active,
            ).group_by(ManualGapSettings.id)

            filter_or = [ManualGapSettings.all_services.is_(True)]
            if schedule_ids:
                filter_or.extend([
                    # если один из schedules_id есть в настройке по графикам
                    self.session.query(ManualGapSettingsSchedules).filter(
                        ManualGapSettingsSchedules.gap_settings_id == ManualGapSettings.id,
                        ManualGapSettingsSchedules.schedule_id.in_(schedule_ids),
                    ).exists(),
                    # или если у этой настройки есть service_id и нет schedules_id
                    self.session.query(ManualGapSettingsServices).join(
                        ManualGapSettingsSchedules,
                        ManualGapSettingsServices.gap_settings_id == ManualGapSettingsSchedules.gap_settings_id,
                        isouter=True,
                    ).filter(
                        ManualGapSettingsServices.gap_settings_id == ManualGapSettings.id,
                        ManualGapSettingsServices.service_id == Member.service_id,
                    ).group_by(
                        ManualGapSettingsServices.id
                    ).having(func.count(ManualGapSettingsSchedules.id) == 0).exists(),
                ])
            else:
                filter_or.extend([
                    self.session.query(ManualGapSettingsServices).filter(
                        ManualGapSettingsServices.gap_settings_id == ManualGapSettings.id,
                        ManualGapSettingsServices.service_id == Member.service_id,
                    ).exists(),
                    self.session.query(ManualGapSettingsSchedules).join(Schedule).filter(
                        ManualGapSettingsSchedules.gap_settings_id == ManualGapSettings.id,
                        Schedule.service_id == Member.service_id,
                    ).exists()
                ])
            query = query.filter(or_(*filter_or))

        return super().filter_objects(query=query)

    @router.post('/', status_code=status.HTTP_201_CREATED)
    def create(self, gap_settings_schema: ManualGapSettingsCreateSchema) -> ManualGapSettingsListSchema:
        self._check_create_permissions(schema=gap_settings_schema)
        self._validate_schema(schema=gap_settings_schema)
        obj = create_gap_settings(db=self.session, staff_id=self.current_user.id, schema=gap_settings_schema)
        update_manual_gaps.delay(gap_settings_id=obj.id)
        return obj

    @router.patch('/{gap_settings_id}')
    def patch(self, gap_settings_id: int, gap_settings_schema: ManualGapSettingsPatchSchema) -> ManualGapSettingsListSchema:
        obj = self.get_object(object_id=gap_settings_id)
        self._validate_schema(schema=gap_settings_schema, obj=obj)
        obj, has_changes = patch_gap_settings(db=self.session, obj=obj, schema=gap_settings_schema)
        if has_changes:
            update_manual_gaps.delay(gap_settings_id=obj.id)
        return obj

    @router.delete('/{gap_settings_id}', status_code=status.HTTP_204_NO_CONTENT)
    def delete(self, gap_settings_id: int):
        obj = self.get_object(gap_settings_id)
        obj.is_active = False
        self.session.commit()
        update_manual_gaps.delay(gap_settings_id=gap_settings_id)

    def _check_create_permissions(self, schema: ManualGapSettingsCreateSchema) -> None:
        if is_superuser(self.current_user):
            return
        if self.current_user.id != schema.staff_id:
            raise PermissionDenied(message={
                'ru': 'Нет разрешения на создание интервала отсутствия',
                'en': 'No permission to create gap interval',
            })

    def _check_change_permissions(self) -> None:
        obj = self.get_object(self.request.path_params['gap_settings_id'])
        if self.current_user.id != obj.staff_id:
            raise PermissionDenied(message={
                'ru': 'Нет разрешения на изменение интервала отсутствия',
                'en': 'No permission to change gap interval',
            })

    def _validate_schema(
        self,
        schema: Union[ManualGapSettingsCreateSchema, ManualGapSettingsPatchSchema],
        obj: Optional[ManualGapSettings] = None,
    ) -> None:
        if schema.schedules is None:
            schedules = obj.schedules if obj else []
        else:
            schedules = self.session.query(Schedule).filter(
                Schedule.id.in_(schema.schedules)
            ).all()
            if len(schedules) != len(schema.schedules):
                raise RecordNotFound(message={
                    'ru': 'Не все указанные расписания существуют',
                    'en': 'Not all specified schedules exist',
                })
        if schema.services is None:
            service_ids = {item.id for item in obj.services} if obj else set()
        else:
            service_ids = schema.services
            count = self.session.query(Service).filter(
                Service.id.in_(schema.services)
            ).count()
            if count != len(service_ids):
                raise RecordNotFound(message={
                    'ru': 'Не все указанные сервисы существуют',
                    'en': 'Not all specified services exist',
                })

        schedule_service_ids = {item.service_id for item in schedules}
        if schedule_service_ids.difference(service_ids):
            raise ScheduleServiceNotInTheList

        if schema.start is not None and localize(schema.start).date() < today():
            raise ManualGapStartInPast

        gap_start = schema.start if schema.start else obj.start
        gap_end = schema.end if schema.end else obj.end
        gap_length = gap_end - gap_start

        max_length = None
        gap_recurrence = schema.recurrence if schema.recurrence else obj.recurrence
        if gap_recurrence == enums.ManualGapRecurrence.day:
            max_length = datetime.timedelta(days=1)
        elif gap_recurrence == enums.ManualGapRecurrence.week:
            max_length = datetime.timedelta(days=7)
        elif gap_recurrence == enums.ManualGapRecurrence.fortnight:
            max_length = datetime.timedelta(days=14)
        elif gap_recurrence == enums.ManualGapRecurrence.month:
            max_length = datetime.timedelta(days=30)

        if gap_length < datetime.timedelta() or (max_length and gap_length > max_length):
            raise InvalidManualGapLength
