import datetime
import logging

from typing import List, Tuple, Dict, Iterable, Optional

from fastapi import status
from fastapi_utils.cbv import cbv
from fastapi_utils.inferring_router import InferringRouter

from sqlalchemy.orm import Query

from watcher.db.composition import Composition
from watcher.crud.rating import get_or_create_ratings_by_schedule_staff_ids
from watcher.config import settings
from watcher import enums
from watcher.api.routes.base import BaseRoute
from watcher.api.schemas.base import CursorPaginationResponse
from watcher.api.schemas.interval import (
    IntervalListSchema,
    IntervalListPutSchema,
    IntervalPutSchema,
)
from watcher.crud.revision import (
    query_current_revision,
    get_revision_by_apply_datetime,
)
from watcher.crud.base import get_object_by_model_or_404
from watcher.crud.composition import get_compositions_by_ids
from watcher.crud.interval import query_intervals_by_revision
from watcher.crud.schedule import get_schedule_or_404
from watcher.crud.role import get_roles_by_ids
from watcher.db import Interval, Slot, Schedule, Revision, from_transfer
from watcher.logic.boundaries_revision import recalculated_fields_changes
from watcher.logic.exceptions import (
    BadRequest,
    PermissionDenied,
    RecalculationInProcess,
    BadRotatingIntervalSettings,
)
from watcher.logic.interval import (
    clone_interval,
    create_interval,
    is_empty,
    check_rotation_validity,
)
from watcher.logic.permissions import (
    is_user_responsible_for_service_or_schedule,
)
from watcher.logic.revision import create_schedule_revision, remove_revision
from watcher.logic.timezone import now
from watcher.tasks.generating_shifts import revision_shift_boundaries
from watcher.tasks.people_allocation import start_people_allocation

logger = logging.getLogger(__name__)
router = InferringRouter()


@cbv(router)
class IntervalRoute(BaseRoute):
    model = Interval
    joined_load = ('slots',)
    joined_load_list = ('slots',)

    def _check_create_permissions(self, schedule: Schedule):
        if not is_user_responsible_for_service_or_schedule(
            db=self.session,
            schedule=schedule,
            staff=self.current_user,
        ):
            raise PermissionDenied(message={
                'ru': 'Нет разрешения на создание интервала',
                'en': 'No permission to create interval',
            })

    def list_objects_by_model(self) -> Query:
        super_filter = super().list_objects_by_model()
        params_keys = ''.join(self.filter_params.keys())
        if 'revision_id' in params_keys:
            return super_filter

        return (
            super_filter.join(Revision, Revision.id == Interval.revision_id)
                .filter(Revision.state == enums.RevisionState.active)
        )

    def _add_rating_for_new_staff(self, compositions: List[Composition], schedule_id: int):
        staff_ids = set()
        for composition in compositions:
            for staff in composition.participants:
                staff_ids.add(staff.id)
        if staff_ids:
            get_or_create_ratings_by_schedule_staff_ids(db=self.session, schedule_ids=[schedule_id],
                                                        staff_ids=staff_ids)

    @router.get('/{interval_id}')
    def retrieve(self, interval_id: int) -> IntervalListSchema:
        return self.get_object(interval_id)

    @router.get('/')
    def list(self) -> CursorPaginationResponse[IntervalListSchema]:
        return self.list_objects()

    @router.put('/', status_code=status.HTTP_200_OK)
    def put(self, intervals_data: IntervalListPutSchema) -> List[IntervalListSchema]:
        schedule = get_schedule_or_404(self.session, intervals_data.schedule_id)
        self._check_create_permissions(schedule=schedule)

        schema_intervals = intervals_data.intervals
        revision_id = intervals_data.revision_id
        if not revision_id:
            # проверим, нет ли ревизии, которая начинается в тот же момент
            old_revision = get_revision_by_apply_datetime(
                db=self.session,
                schedule_id=intervals_data.schedule_id,
                apply_datetime=intervals_data.apply_datetime,
            )
        else:
            old_revision = get_object_by_model_or_404(
                db=self.session,
                model=Revision,
                object_id=revision_id,
                joined_load=('next', 'prev',)
            )

        logger.info(
            f'Creating new revision for schedule :{intervals_data.schedule_id} '
            f'replacing revision: {revision_id}, apply_datetime: {intervals_data.apply_datetime}'
        )

        current_revision = query_current_revision(
            db=self.session, schedule_id=schedule.id
        )
        composition_ids = [s.composition_id for i in intervals_data.intervals for s in i.slots]
        compositions = get_compositions_by_ids(db=self.session, composition_ids=composition_ids)
        self._validate(
            schedule=schedule,
            intervals_data=intervals_data,
            revision_id=revision_id,
            current_revision=current_revision,
            compositions=compositions
        )
        if not intervals_data.apply_datetime:
            intervals_data.apply_datetime = now()

        has_changes, need_revision_shift = self._check_boundaries_recalculate(
            schema_intervals=schema_intervals,
            old_revision=old_revision,
            apply_datetime=intervals_data.apply_datetime,
        )

        revision = create_schedule_revision(
            db=self.session,
            schedule_id=schedule.id,
            apply_datetime=intervals_data.apply_datetime,
        )
        if old_revision:
            # редактируют существующую ревизию, нужно ее убрать
            remove_revision(revision=old_revision)
        intervals = self._create_intervals(
            revision=revision,
            schema_intervals=schema_intervals,
        )
        self._add_rating_for_new_staff(compositions=compositions, schedule_id=intervals_data.schedule_id)
        self.session.commit()
        self._send_recalculation(
            schedule=schedule,
            has_changes=has_changes,
            need_revision_shift=need_revision_shift,
            apply_datetime=revision.apply_datetime,
            revision_id=revision.id,
        )
        return sorted(intervals, key=lambda x: x.order)

    def _send_recalculation(
        self, schedule: Schedule, has_changes: bool,
        need_revision_shift: bool, revision_id: int,
        apply_datetime: datetime.datetime,
    ) -> None:
        """ Запускает нужную таску в зависимости от того, нужно ли делать перерасчет границ """

        if has_changes:
            if schedule.state != enums.ScheduleState.active:
                logger.info(f'Skipping sending tasks for not active schedule {schedule.id}')
                return
            if schedule.recalculate is False:
                logger.info(f'Skipping sending tasks for not recalculated schedule {schedule.id}')
                return
            schedule.recalculation_in_process = True
            if need_revision_shift:
                logger.info(f'Sending revision_shift_boundaries for {schedule.id}, revision: {revision_id}')
                revision_shift_boundaries.delay(
                    schedule_id=schedule.id,
                    revision_id=revision_id,
                )
            else:
                logger.info(f'Sending start_people_allocation for {schedule.id}, group {schedule.schedules_group_id}')
                kwargs = {
                    settings.FORCE_TASK_DELAY: True,
                    'schedules_group_id': schedule.schedules_group_id,
                    'start_date': apply_datetime,
                }
                start_people_allocation.delay(**kwargs)

    def _clone_intervals(
        self,
        intervals: List[Interval],
        ids_to_exclude: Optional[Iterable[int]] = None,
    ) -> Tuple[Dict[int, Interval], Dict[int, Dict[int, Slot]]]:
        """
        :param intervals: схема объектов
        :return:
            - копии переданных интервалов в формате {prev_id1: copy1, prev_id2: copy2...}
        (словарь, где ключи - id старых интервалов, а значения - новые копии этих интервалов)
            - копии переданных слотов в формате {prev_interval_id1: {prev_slot_id1: copy_slot1...}...}
        (словарь, где ключи - id старых интервалов, которому принадлежали слоты, а значения - другие словари
            где ключи - id старых слотов, занчения - новые копии этих слотов)
        """
        intervals_copy = dict()
        slots_copy = dict()

        for interval_to_copy in intervals:
            if ids_to_exclude and interval_to_copy.id in ids_to_exclude:
                continue
            interval, slots = clone_interval(interval_to_copy)
            intervals_copy[interval_to_copy.id] = interval
            slots_copy[interval_to_copy.id] = slots

        return intervals_copy, slots_copy

    def _create_intervals(
        self,
        revision: Revision,
        schema_intervals: Iterable[IntervalPutSchema],
    ) -> List[Interval]:
        new_intervals = [
            create_interval(interval_schema, revision=revision)
            for interval_schema in schema_intervals
        ]
        self.session.add_all(new_intervals)

        return new_intervals

    def _check_boundaries_recalculate(
        self,
        schema_intervals: List[IntervalPutSchema],
        old_revision: Optional[Revision],
        apply_datetime: datetime.datetime,
    ) -> Tuple[bool, bool]:
        """
        Если будет удаление/добавление существующих слотов/интервалов, то нужно запускать перераспределение людей

        Если изменились параметры интервала, нужно запустить таску на перерасчет границ
        Если изменились только соствы слотов, то нужно запустить таску перераспределение людей

        :returns has_changes - есть изменения которые требуют перераспределения людей или пересчет границ
                 need_revision_shift - нужно ли делать именно пересчет границ
        """
        db_intervals = []
        if old_revision:
            if old_revision.apply_datetime != apply_datetime:
                return True, True
            db_intervals = query_intervals_by_revision(db=self.session, revision_id=old_revision.id).all()

        db_intervals.sort(key=lambda x: x.id)
        schema_intervals.sort(key=lambda x: x.id if x.id else -1)

        if list(map(lambda x: x.id, db_intervals)) != list(map(lambda x: x.id, schema_intervals)):
            return True, True

        has_changes = need_revision_shift = False

        for interval, schema in zip(db_intervals, schema_intervals):
            if len(interval.slots) != len(schema.slots):
                has_changes = need_revision_shift = True
            else:
                has_changes, need_revision_shift = recalculated_fields_changes(interval, schema)

            if has_changes:
                break

        return has_changes, need_revision_shift

    def _validate(
        self,
        schedule: Schedule,
        intervals_data: IntervalListPutSchema,
        revision_id: int,
        current_revision: Optional[Revision],
        compositions: List[Composition]
    ) -> None:
        """
        Валидируем входные данные

        :param schedule: расписание которому должны принадлежать интервалы
        :param intervals: спискок схем интервалов
        :return:
        """
        if schedule.recalculation_in_process:
            raise RecalculationInProcess()

        if revision_id:
            revision_from_request = get_object_by_model_or_404(
                db=self.session, model=Revision, object_id=revision_id,
            )

            if revision_from_request.apply_datetime <= now():
                raise BadRequest(message={
                    'ru': 'Можно редактировать только будущие ревизии',
                    'en': 'Only future revisions can be edited',
                })

        if (
            current_revision
            and intervals_data.apply_datetime
            and intervals_data.apply_datetime < current_revision.apply_datetime
        ):
            raise BadRequest(message={
                'ru': 'Ревизия не может начинаться раньше чем текущая',
                'en': 'Revision cannot start earlier than current revision',
            })

        backup_is_next_primary = False
        shared_composition_id = None

        for interval in intervals_data.intervals:
            if interval.primary_rotation:
                if schedule.rotation != enums.IntervalRotation.default:
                    raise BadRequest(message={
                        'ru': 'Конфликт настроек ротации расписания и отдельного интервала',
                        'en': 'Conflict of schedule and individual interval rotation settings'
                    })

        for interval in intervals_data.intervals:
            if schedule.id != interval.schedule_id:
                raise BadRequest(message={
                    'ru': 'Все интервалы должны принадлежать заданному расписанию',
                    'en': 'All intervals must belong to the specified schedule'
                })

            if interval.type_employment == enums.IntervalTypeEmployment.full and not interval.slots:
                raise BadRequest(message={
                    'ru': 'У не пустых интервалов должны быть слоты',
                    'en': 'Non-empty intervals must have slots'
                })

            if is_empty(interval) and interval.slots:
                raise BadRequest(message={
                    'ru': 'У пустых интервалов не должно быть слотов',
                    'en': 'Empty intervals should not have slots'
                })

            if interval.primary_rotation:
                backup_is_next_primary = True
                shared_composition_id = None
            elif schedule.rotation is enums.IntervalRotation.backup_is_next_primary:
                # переносим настройку с расписания на интервал (для обратной совместимости)
                interval.primary_rotation = True

            if backup_is_next_primary or not (schedule.rotation is enums.IntervalRotation.default):
                if not shared_composition_id:
                    shared_composition_id = interval.slots[0].composition_id
                if not check_rotation_validity(
                    interval=interval, expected_composition_id=shared_composition_id
                ):
                    raise BadRotatingIntervalSettings()

            if any(slot.interval_id and slot.interval_id != interval.id for slot in interval.slots):
                raise BadRequest(message={
                    'ru': 'Переданные слоты в интервалах должны иметь соответствующий interval_id',
                    'en': 'Slots in the intervals must have the corresponding interval_id'
                })
            if interval.backup_takes_primary_shift and all(slot.is_primary for slot in interval.slots):
                raise BadRequest(message={
                    'ru': 'С настройкой backup_takes_primary_shift должен быть передан хотя бы один backup слот',
                    'en': 'At least one backup slot must be provided in presence of backup_takes_primary_shift option',
                })

            if interval.primary_rotation:
                backup_is_next_primary = False

        role_ids = [s.role_on_duty_id for i in intervals_data.intervals for s in i.slots]
        roles = get_roles_by_ids(db=self.session, role_ids=role_ids)
        if (
            any(role.code in from_transfer.Role.CAN_NOT_USE_FOR_DUTY for role in roles)
            or any(role.scope.can_issue_at_duty_time is False for role in roles)
        ):
            raise BadRequest(message={
                'ru': 'Указанную роль нельзя выдать на дежурство',
                'en': 'Specified role cannot be assigned to duty'
            })

        if any(role.service_id != schedule.service_id for role in roles if role.service_id):
            raise BadRequest(message={
                'ru': 'Указанная роль не принадлежит данному сервису',
                'en': 'Specified role does not correspond to this service'
            })

        if any(composition.service_id != schedule.service_id for composition in compositions):
            raise BadRequest(message={
                'ru': 'Указанная композиция не соответствует данному сервису',
                'en': 'Specified composition does not correspond to this service'
            })
