from fastapi_utils.cbv import cbv
from fastapi_utils.inferring_router import InferringRouter

from watcher.logic.exceptions import PermissionDenied
from watcher.logic.service import is_responsible_in_service
from watcher.logic.problem import resolve_problem
from watcher.logic.permissions import is_superuser
from watcher.api.schemas.base import CursorPaginationResponse
from watcher.api.schemas.problem import ProblemListSchema
from watcher.db import Problem

from .base import BaseRoute

router = InferringRouter()


@cbv(router)
class ProblemRoute(BaseRoute):
    model = Problem
    joined_load = (
        'duty_gap', 'manual_gap',
        'manual_gap.gap_settings',
        'staff', 'shift',
    )
    joined_load_list = (
        'duty_gap', 'manual_gap',
        'manual_gap.gap_settings',
        'staff', 'shift',
    )

    @router.get('/{problem_id}')
    def retrieve(self, problem_id: int) -> ProblemListSchema:
        return self.get_object(object_id=problem_id)

    @router.get('/')
    def list(self) -> CursorPaginationResponse[ProblemListSchema]:
        return self.list_objects()

    @router.post('/{problem_id}/resolve')
    def post(self, problem_id: int) -> ProblemListSchema:
        obj = self.get_object(
            object_id=problem_id,
            extra_joined=('shift.schedule', )
        )
        if not is_superuser(staff=self.current_user):
            if not is_responsible_in_service(
                db=self.session, staff_id=self.current_user.id,
                service_id=obj.shift.schedule.service_id
            ):
                raise PermissionDenied(message={
                    'ru': 'Нет прав на изменение статуса проблемы',
                    'en': 'No permission to change problem status',
                })
        resolve_problem(obj)
        return obj
