import logging

from fastapi import status
from fastapi_utils.cbv import cbv
from fastapi_utils.inferring_router import InferringRouter
from sqlalchemy.orm import Query

from watcher.api.schemas.base import CursorPaginationResponse
from watcher.api.schemas.revision import (
    RevisionListSchema
)
from watcher.crud.revision import get_current_revision
from watcher.db import Revision, Shift
from watcher.logic.permissions import is_user_responsible_for_service_or_schedule
from watcher.logic.exceptions import (
    BadRequest,
    PermissionDenied,
    RecalculationInProcess,
)
from watcher.logic.timezone import now, localize
from watcher.logic.revision import remove_revision
from watcher.tasks.generating_shifts import revision_shift_boundaries
from watcher.crud.shift import (
    query_shifts_by_revision,
)
from watcher import enums

from .base import BaseRoute

logger = logging.getLogger(__name__)
router = InferringRouter()


@cbv(router)
class RevisionRoute(BaseRoute):
    model = Revision
    joined_load = ('intervals', 'next', 'prev', 'schedule')

    def _check_change_permissions(self):
        revision = self.get_object(self.request.path_params['revision_id'])
        if not is_user_responsible_for_service_or_schedule(
            db=self.session,
            schedule=revision.schedule,
            staff=self.current_user,
        ):
            raise PermissionDenied(message={
                'ru': 'Нет разрешения на изменение ревизии',
                'en': 'No permission to change revision',
            })

    @router.get('/current-revision')
    def current_revision(self, schedule_id: int) -> RevisionListSchema:
        return get_current_revision(db=self.session, schedule_id=schedule_id)

    @router.get('/{revision_id}')
    def retrieve(self, revision_id: int) -> RevisionListSchema:
        obj = self.get_object(object_id=revision_id)

        if obj.prev:
            obj.prev_id = obj.prev.id

        return obj

    def list_objects_by_model(self) -> Query:
        query = super().list_objects_by_model()
        params_keys = ''.join(self.filter_params.keys())
        if 'state' in params_keys:
            return query

        return query.filter(Revision.state == enums.RevisionState.active)

    @router.get('/')
    def list(self) -> CursorPaginationResponse[RevisionListSchema]:
        return self.list_objects()

    @router.delete('/{revision_id}', status_code=status.HTTP_204_NO_CONTENT)
    def destroy(self, revision_id: int):
        revision = self.get_object(object_id=revision_id)

        if revision.schedule.recalculation_in_process:
            raise RecalculationInProcess()

        if localize(revision.apply_datetime) <= now():
            raise BadRequest(message={
                'ru': 'Можно деактивировать только будущие ревизии',
                'en': 'Only future revisions can be disabled'
            })
        prev_revision = revision.prev
        remove_revision(revision=revision)

        date_from = revision.apply_datetime
        if prev_revision:
            # если была предыдущая ревизия - значит смену могли обрезать
            # и нужно запустить пересчет с ее начала, чтобы она удалились
            # и создалась потом целиковой
            prev_shift = query_shifts_by_revision(
                db=self.session,
                revision_id=prev_revision.id
            ).order_by(Shift.end.desc()).first()
            if prev_shift:
                date_from = prev_shift.start
        logger.info(f'Sending revision_shift_boundaries for schedule: {revision.schedule_id}, from: {date_from}')

        revision_shift_boundaries.delay(
            schedule_id=revision.schedule_id,
            date_from=date_from,
        )
