from typing import Optional, List

from fastapi import status
from fastapi_utils.cbv import cbv
from fastapi_utils.inferring_router import InferringRouter
from sqlalchemy.orm import Query, joinedload

from watcher.config import settings
from watcher.logic.clients.abc import abc_client
from watcher.api.schemas.schedule import (
    ScheduleWithValidatorSchema,
    ScheduleListSchema,
    ScheduleCreateSchema,
    SchedulePatchSchema,
)
from watcher.api.schemas.base import CursorPaginationResponse
from watcher.crud.schedule_group import get_schedules_group_by_id
from watcher.tasks.generating_shifts import (
    delete_disabled_shifts,
    proceed_new_shifts,
)
from watcher.tasks.people_allocation import start_people_allocation
from watcher.tasks.manual_gap import update_manual_gaps
from watcher.logic.timezone import now
from watcher.crud.base import get_object_by_model_or_404
from watcher.db import (
    Schedule, Service, Interval, Slot, Revision, ManualGapSettings
)
from watcher.logic.exceptions import (
    IsAutoGeneratedGroup,
    SlugMustBeUnique,
    ServiceMustBeActive,
    PermissionDenied,
    NotResponsibleInAllServices,
    BadRotatingIntervalSettings,
)
from watcher.crud.schedule import create_schedule, patch_schedule
from watcher.logic.permissions import (
    is_user_responsible_for_service_or_schedule,
    is_user_in_service_or_responsible,
    services_responsible_in,
)
from watcher.crud.revision import (
    query_first_revisions_by_schedules,
    query_revision_by_schedule,
)
from watcher.crud.manual_gap import (
    query_affected_by_schedule_deletion_gap_settings
)
from watcher.logic.schedule import get_composition_id_for_rotation
from watcher.logic.interval import check_rotation_validity
from watcher import enums
from .base import BaseRoute

router = InferringRouter()


@cbv(router)
class ScheduleRoute(BaseRoute):
    model = Schedule
    joined_load = (
        'show_in_services', 'schedules_group', 'service',
        'actual_problems', 'actual_problems.duty_gap', 'actual_problems.manual_gap',
    )
    joined_load_list = (
        'show_in_services', 'schedules_group', 'service', 'actual_problems',
        'actual_problems.duty_gap', 'actual_problems.manual_gap',
    )

    def _check_create_permissions(self, service_id: int):
        if not is_user_in_service_or_responsible(
            db=self.session,
            staff=self.current_user,
            service_id=service_id,
        ):
            raise PermissionDenied(message={
                'ru': 'Нет разрешения на создание расписания',
                'en': 'No permission to create schedule'
            })

    def _check_change_permissions(self):
        schedule = self.get_object(self.request.path_params['schedule_id'])
        if not is_user_responsible_for_service_or_schedule(
            db=self.session,
            schedule=schedule,
            staff=self.current_user,
        ):
            raise PermissionDenied(message={
                'ru': 'Нет разрешения на изменение расписания',
                'en': 'No permission to change schedule',
            })

    def _check_recalculate_permissions(self, schedule: Schedule):
        if not is_user_responsible_for_service_or_schedule(
            db=self.session,
            schedule=schedule,
            staff=self.current_user,
        ):
            raise PermissionDenied(message={
                'ru': 'Нет разрешения на пересчет расписания',
                'en': 'No permission to recalculate schedule'
            })

    @router.get('/{schedule_id}')
    def retrieve(self, schedule_id: int) -> ScheduleListSchema:
        schedule = self.get_object(object_id=schedule_id)
        self._add_revision(items=[schedule])
        self._add_show_in_staff(items=[schedule])
        self.add_is_responsible(items=[schedule])
        return schedule

    @router.delete('/{schedule_id}', status_code=status.HTTP_204_NO_CONTENT)
    def destroy(self, schedule_id: int):
        obj = self.get_object(object_id=schedule_id)
        self._delete_affected_gap_settings(schedule_id=obj.id)
        self.session.query(Schedule).filter(Schedule.id == obj.id).delete()
        self.session.commit()

    @router.patch('/{schedule_id}')
    def patch(self, schedule_id: int, schedule: SchedulePatchSchema) -> ScheduleListSchema:
        obj = self.get_object(object_id=schedule_id, extra_joined=('intervals', 'intervals.slots'))
        saved_schedule_rotation = obj.rotation
        self._validate_schema(schema=schedule, obj=obj)
        initial_state = obj.state
        patched_object = patch_schedule(
            db=self.session, obj=obj, schema=schedule,
        )
        if initial_state != patched_object.state:
            if patched_object.state == enums.ScheduleState.disabled:
                delete_disabled_shifts.delay(schedule_id=schedule_id)
            elif patched_object.state == enums.ScheduleState.active:
                proceed_new_shifts.delay(schedule_id=schedule_id)
        if obj.rotation != saved_schedule_rotation:
            self._reallocate(obj=obj)
        self.add_is_responsible(items=[patched_object])
        return patched_object

    @router.get('/')
    def list(self) -> CursorPaginationResponse[ScheduleListSchema]:
        pagination_response = self.list_objects()
        self._add_revision(items=pagination_response.result)
        self._add_show_in_staff(items=pagination_response.result)
        self.add_is_responsible(items=pagination_response.result)
        return pagination_response

    @router.post('/', status_code=status.HTTP_201_CREATED)
    def create(self, schedule: ScheduleCreateSchema) -> ScheduleListSchema:
        self._check_create_permissions(service_id=schedule.service_id)

        self._validate_schema(schema=schedule)
        self._validate_service_state(schedule.service_id)
        self._validate_unique_slug(
            service_id=schedule.service_id,
            slug=schedule.slug,
        )
        schedule_obj = create_schedule(
            db=self.session,
            schedule=schedule,
            author=self.current_user,
        )
        self.add_is_responsible(items=[schedule_obj])
        return schedule_obj

    @router.post('/{schedule_id}/recalculate')
    def recalculate(self, schedule_id: int) -> ScheduleListSchema:
        schedule = self.get_object(object_id=schedule_id)
        self._check_recalculate_permissions(schedule=schedule)
        self._reallocate(obj=schedule)
        self._add_revision(items=[schedule])
        self._add_show_in_staff(items=[schedule])
        self.add_is_responsible(items=[schedule])
        return schedule

    def filter_objects(self, query: Query) -> Query:
        if 'has_revision' in self.request.query_params:
            if str.lower(self.request.query_params['has_revision']) in ('true', '1'):
                query = query.filter(
                    query_revision_by_schedule(
                        db=self.session,
                        schedule_id=Schedule.id,
                    ).exists()
                )
        return super().filter_objects(query=query)

    def _validate_service_state(self, service_id: int) -> None:
        service = get_object_by_model_or_404(
            db=self.session, model=Service,
            object_id=service_id,
        )
        if service.state in (enums.ServiceState.closed, enums.ServiceState.deleted):
            raise ServiceMustBeActive

    def _check_schedules_group_id(self, schedules_group_id: int) -> None:
        if not schedules_group_id:
            return

        schedules_group = get_schedules_group_by_id(self.session, schedules_group_id)

        if schedules_group and schedules_group.autogenerated:
            raise IsAutoGeneratedGroup()

    def _validate_schema(self, schema: ScheduleWithValidatorSchema, obj: Optional[Schedule] = None) -> None:
        self._check_schedules_group_id(schema.schedules_group_id)

        if schema.show_in_services:
            # проверяем, что пользователь может управлять во всех переданных сервисах
            services_ids = schema.show_in_services
            if obj:
                services_ids = {
                    service_id for service_id in services_ids
                    if service_id not in {
                        service.id for service in obj.show_in_services
                    }
                }
            responsible_in_services = services_responsible_in(
                db=self.session,
                services_ids=services_ids,
                staff=self.current_user,
            )
            if services_ids > responsible_in_services:
                raise NotResponsibleInAllServices()

        rotation = schema.rotation
        if not rotation:
            rotation = enums.IntervalRotation.default

        if obj and rotation != enums.IntervalRotation.default:
            composition_id = get_composition_id_for_rotation(schedule=obj)
            if composition_id:
                for interval in obj.intervals:
                    if not check_rotation_validity(
                        interval=interval, expected_composition_id=composition_id
                    ):
                        raise BadRotatingIntervalSettings()

    def _validate_unique_slug(self, service_id: int, slug: str) -> None:
        watcher_schedules = self.session.query(Schedule).filter(
            Schedule.slug == slug,
            Schedule.service_id == service_id,
        ).count()
        if watcher_schedules:
            raise SlugMustBeUnique

        abc_schedules = abc_client.get_schedules(service_id=service_id)
        for abc_schedule in abc_schedules:
            if abc_schedule['slug'] == slug:
                raise SlugMustBeUnique

    def _add_revision(self, items: List[Schedule]) -> None:
        if 'revision' in self.fields_params:
            revisions = query_first_revisions_by_schedules(
                db=self.session,
                schedules_ids=[
                    schedule.id for schedule
                    in items
                ]
            ).all()
            revisions_map = {
                revision.schedule_id: revision
                for revision in revisions
            }
            for item in items:
                setattr(item, 'revision', revisions_map.get(item.id))

    def _add_show_in_staff(self, items: List[Schedule]) -> None:
        if 'show_in_staff' in self.fields_params:
            slots_with_showed_staff = (
                self.session.query(Slot)
                .join(Interval, Interval.id == Slot.interval_id)
                .join(Revision, Revision.id == Interval.revision_id)
                .options(
                    joinedload(Slot.interval),
                )
                .filter(
                    ~Slot.show_in_staff,
                    Interval.type_employment == enums.IntervalTypeEmployment.full,
                    Interval.schedule_id.in_([schedule.id for schedule in items]),
                    Revision.state == enums.RevisionState.active,
                    ~Revision.next.has(),
                )
            ).all()
            show_in_staff_map = {
                slot.interval.schedule_id: False
                for slot in slots_with_showed_staff
            }
            for item in items:
                setattr(item, 'show_in_staff', show_in_staff_map.get(item.id, True))

    def _delete_affected_gap_settings(self, schedule_id: int):
        related_gap_settings = query_affected_by_schedule_deletion_gap_settings(
            db=self.session, schedule_id=schedule_id,
        ).all()
        obj_ids = {obj.id for obj in related_gap_settings}

        self.session.query(ManualGapSettings).filter(
            ManualGapSettings.id.in_(obj_ids)
        ).update({ManualGapSettings.is_active: False}, synchronize_session=False,)

        for obj_id in obj_ids:
            update_manual_gaps.delay(gap_settings_id=obj_id)

    def _reallocate(self, obj: Schedule):
        obj.recalculation_in_process = True
        kwargs = {
            settings.FORCE_TASK_DELAY: True,
            'schedules_group_id': obj.schedules_group_id,
            'start_date': now(),
        }
        start_people_allocation.delay(**kwargs)
