from fastapi import status
from fastapi_utils.cbv import cbv
from fastapi_utils.inferring_router import InferringRouter
from sqlalchemy.orm import Query
from typing import Optional, Union

from watcher.api.schemas.base import CursorPaginationResponse
from watcher.api.schemas.schedulegroup import (
    ScheduleGroupListSchema,
    ScheduleGroupPatchSchema,
    ScheduleGroupCreateSchema,
    ScheduleGroupResponsiblePutSchema,
    ScheduleGroupSchedulesPutSchema,
)
from watcher.crud.schedule_group import (
    create_schedules_group,
    schedules_group_validate_destroy,
)
from watcher.db import SchedulesGroup
from watcher.logic.exceptions import (
    ScheduleGroupHasReference,
    PermissionDenied,
    NotResponsibleInAllServices,
    GroupSlugMustBeUnique,
)
from watcher.logic.exceptions import (
    IsAutoGeneratedGroup
)
from watcher.logic.schedule_group import create_autogenerated_group_object

from watcher.db import (
    ScheduleGroupResponsible,
    Schedule,
)
from watcher.crud.base import (
    update_many_to_many_for_field,
)
from watcher.logic.permissions import (
    is_group_responsible,
    services_responsible_in,
)
from .base import BaseRoute

router = InferringRouter()


@cbv(router)
class ScheduleGroupRoute(BaseRoute):
    model = SchedulesGroup
    is_responsible: Optional[bool] = None
    joined_load = ('responsibles', 'schedules', )
    joined_load_list = ('responsibles', 'schedules', )

    def _check_change_permissions(self):
        """
        Вносить изменения в группу может любой из
        ответственных
        """
        group_id = self.request.path_params['schedule_group_id']
        if not is_group_responsible(db=self.session, group_id=group_id, staff=self.current_user):
            raise PermissionDenied(message={
                'ru': 'Пользователь не является ответственным за группу',
                'en': 'User is not one of group responsibles',
            })

    def list_objects_by_model(self) -> Query:
        return (
            super().list_objects_by_model()
            .filter(~SchedulesGroup.autogenerated)
        )

    def _validate_schema(
        self,
        schema: Union[ScheduleGroupCreateSchema, ScheduleGroupPatchSchema],
        obj: Optional[SchedulesGroup] = None,
    ) -> None:
        if schema.slug and (obj and obj.slug != schema.slug or not obj):
            groups_with_same_slug = self.session.query(SchedulesGroup).filter(
                SchedulesGroup.slug == schema.slug,
            )
            if groups_with_same_slug.count():
                raise GroupSlugMustBeUnique

    def filter_objects(self, query: Query) -> Query:
        """
        Фильтруем группы по переданным фильтрам
        """
        if self.is_responsible:
            query = query.filter(
                SchedulesGroup.responsibles.any(
                    ScheduleGroupResponsible.responsible_id == self.current_user.id
                )
            )
        return super().filter_objects(query=query)

    @router.get('/{schedule_group_id}')
    def retrieve(self, schedule_group_id: int) -> ScheduleGroupListSchema:
        obj = self.get_object(object_id=schedule_group_id)
        self.add_is_responsible(field_param='schedules.is_responsible', items=obj.schedules)
        return obj

    @router.delete('/{schedule_group_id}', status_code=status.HTTP_204_NO_CONTENT)
    def destroy(self, schedule_group_id: int):
        obj = self.get_object(object_id=schedule_group_id)

        if not schedules_group_validate_destroy(self.session, obj.id):
            raise ScheduleGroupHasReference()

        self.session.delete(obj)
        self.session.commit()

    @router.patch('/{schedule_group_id}')
    def patch(self, schedule_group_id: int, schedulegroup: ScheduleGroupPatchSchema) -> ScheduleGroupListSchema:
        obj = self.get_object(object_id=schedule_group_id)
        self._validate_schema(obj=obj, schema=schedulegroup)
        patched_object = self.patch_object(obj=obj, schema=schedulegroup)
        self.add_is_responsible(field_param='schedules.is_responsible', items=patched_object.schedules)
        return patched_object

    @router.get('/')
    def list(self) -> CursorPaginationResponse[ScheduleGroupListSchema]:
        pagination_response = self.list_objects()
        self.add_is_responsible(
            field_param='schedules.is_responsible',
            items=[
                schedule for schedule_group in
                pagination_response.result
                for schedule in schedule_group.schedules
            ]
        )
        return pagination_response

    @router.post('/', status_code=status.HTTP_201_CREATED)
    def create(self, schedulegroup: ScheduleGroupCreateSchema) -> ScheduleGroupListSchema:
        self._validate_schema(schema=schedulegroup)
        schedulegroup_obj = create_schedules_group(
            db=self.session,
            schedulegroup=schedulegroup,
            author=self.current_user,
        )
        return schedulegroup_obj

    @router.put('/{schedule_group_id}')
    def put(self, schedule_group_id: int, responsibles: ScheduleGroupResponsiblePutSchema) -> ScheduleGroupListSchema:
        staff_ids = responsibles.staff

        obj = self.get_object(object_id=schedule_group_id)
        if obj.autogenerated:
            raise IsAutoGeneratedGroup()

        current_refs = self.session.query(ScheduleGroupResponsible).filter(
            ScheduleGroupResponsible.schedule_group_id == schedule_group_id).all()
        current_values = {ref.responsible_id for ref in current_refs}

        update_many_to_many_for_field(
            db=self.session, current=current_values, target=staff_ids,
            table=ScheduleGroupResponsible,
            current_refs=current_refs,
            field_key='responsible_id', obj=obj, related_field='schedule_group_id',
        )
        self.session.refresh(obj)
        self.add_is_responsible(field_param='schedules.is_responsible', items=obj.schedules)
        return obj

    @router.put('/{schedule_group_id}/schedules')
    def put_schedules(self, schedule_group_id: int, schedules: ScheduleGroupSchedulesPutSchema) -> ScheduleGroupListSchema:
        schedule_ids = schedules.schedule_ids
        obj = self.get_object(object_id=schedule_group_id)
        if obj.autogenerated:
            raise IsAutoGeneratedGroup()

        current_schedule_ids = {schedule.id for schedule in obj.schedules}
        remove_schedule_ids = current_schedule_ids - schedule_ids
        new_schedule_ids = schedule_ids - current_schedule_ids

        new_schedules = self.session.query(Schedule).filter(Schedule.id.in_(new_schedule_ids)).all()
        service_ids = {schedule.service_id for schedule in new_schedules}
        responsible_in_services = services_responsible_in(
            db=self.session,
            services_ids=service_ids,
            staff=self.current_user,
        )
        if service_ids > responsible_in_services:
            raise NotResponsibleInAllServices()

        remove_schedules = self.session.query(Schedule).filter(Schedule.id.in_(remove_schedule_ids)).all()
        for schedule in remove_schedules:
            schedule.schedules_group = create_autogenerated_group_object()

        obj.schedules.extend(new_schedules)
        self.session.commit()
        self.session.refresh(obj)
        self.add_is_responsible(field_param='schedules.is_responsible', items=obj.schedules)
        return obj
