from fastapi import status
from fastapi_utils.cbv import cbv
from fastapi_utils.inferring_router import InferringRouter

from watcher.logic.service import is_responsible_in_service
from watcher.api.routes.base import BaseRoute
from watcher.config import settings
from watcher.crud.composition import query_compositions_by_services
from watcher.crud.schedule import query_schedules_by_service
from watcher.crud.shift import query_current_shifts_by_service
from watcher.db import Service, ScheduleShowInServices
from watcher.logic.exceptions import BadRequest, PermissionDenied
from watcher.tasks.shift import finish_shift
from watcher.api.schemas.service import RelatedSchedulesSchema
from watcher.crud.base import update_many_to_many_for_field

delete_all_about_duty_router = InferringRouter()
router = InferringRouter()


@cbv(delete_all_about_duty_router)
class DeleteAllAboutDutyRoute(BaseRoute):
    model = Service

    @delete_all_about_duty_router.delete('/{service_id}', status_code=status.HTTP_204_NO_CONTENT)
    def destroy(self, service_id: int):
        if settings.ENV_TYPE not in [
            settings.ENV_TYPE_DEVELOPMENT,
            settings.ENV_TYPE_TESTING
        ]:
            raise BadRequest(
                message={
                    'ru': 'В продакшн окружении нельзя использовать эту ручку',
                    'en': 'You cannot use this router in a production environment',
                }
            )

        # кидает исключение, если нет такого сервиса
        self.get_object(object_id=service_id)

        # найдем в сервисе все текущие шифты и закончим их
        current_shifts = [
            shift.id for shift in
            query_current_shifts_by_service(
                db=self.session, service_id=service_id
            ).all()
        ]

        for shift_id in current_shifts:
            finish_shift(session=self.session, shift_id=shift_id, _lock=False)

        schedules = query_schedules_by_service(db=self.session, service_id=service_id)
        schedules.delete(synchronize_session=False)

        compositions = query_compositions_by_services(db=self.session, service_ids=[service_id])
        compositions.delete(synchronize_session=False)

        self.session.commit()


@cbv(router)
class ServiceRoute(BaseRoute):
    model = Service

    @router.put('/{service_id}/related_schedules/', status_code=status.HTTP_204_NO_CONTENT)
    def put(self, service_id: int, schema: RelatedSchedulesSchema):
        obj = self.get_object(object_id=service_id)
        if not is_responsible_in_service(
            db=self.session,
            staff_id=self.current_user.id,
            service_id=obj.id
        ):
            raise PermissionDenied(message={
                'ru': 'Нет прав на управление сервисом',
                'en': 'No permission to manage service',
            })

        current_refs = self.session.query(ScheduleShowInServices).filter(
            ScheduleShowInServices.service_id == obj.id
        )
        current_values = {ref.schedule_id for ref in current_refs}

        update_many_to_many_for_field(
            db=self.session, current=current_values,
            target=schema.related_schedules,
            table=ScheduleShowInServices,
            current_refs=current_refs,
            field_key='schedule_id', obj=obj,
            related_field='service_id',
        )
