from fastapi_utils.cbv import cbv
from fastapi_utils.inferring_router import InferringRouter

from watcher.crud.role import get_role_by_id
from watcher.enums import RevisionState
from watcher.api.routes.base import BaseRoute
from watcher.api.schemas.slot import (
    SlotListSchema,
    SlotPatchSchema,
)

from watcher.logic.exceptions import (
    PermissionDenied,
    BadRequest,
)
from watcher.db import (
    Slot,
    Schedule,
    from_transfer,
)
from watcher.logic.permissions import (
    is_user_responsible_for_service_or_schedule,
)

router = InferringRouter()


@cbv(router)
class SlotRoute(BaseRoute):
    model = Slot
    joined_load = ('interval', 'interval.schedule', 'interval.revision', )

    def validate(self, slot: Slot, schema: SlotPatchSchema) -> None:
        if schema.role_on_duty_id:
            role = get_role_by_id(db=self.session, role_id=schema.role_on_duty_id)
            if role is None:
                raise BadRequest(message={
                    'ru': 'Указанная роль не была найдена',
                    'en': 'The specified role was not found'
                })
            if (
                role.code in from_transfer.Role.CAN_NOT_USE_FOR_DUTY
                or role.scope.can_issue_at_duty_time is False
            ):
                raise BadRequest(message={
                    'ru': 'Указанную роль нельзя выдать на дежурство',
                    'en': 'Specified role cannot be assigned to duty'
                })

            if role.service_id and role.service_id != slot.interval.schedule.service_id:
                raise BadRequest(message={
                    'ru': 'Указанная роль не принадлежит данному сервису',
                    'en': 'Specified role does not correspond to this service'
                })

        if slot.interval.revision.state == RevisionState.disabled:
            raise BadRequest(message={
                'ru': 'Редактирование disabled ревизии запрещено',
                'en': 'Editing a disabled revision is prohibited',
            })

    def _check_users_permissions(self, schedule: Schedule):
        if not is_user_responsible_for_service_or_schedule(
            db=self.session,
            schedule=schedule,
            staff=self.current_user,
        ):
            raise PermissionDenied(message={
                'ru': 'У пользователя нет прав на изменения слота',
                'en': 'The user does not have rights to change the slot',
            })

    @router.patch('/{slot_id}')
    def patch(self, slot_id: int, slot: SlotPatchSchema) -> SlotListSchema:
        db_obj = self.get_object(object_id=slot_id)
        self._check_users_permissions(db_obj.interval.schedule)
        self.validate(slot=db_obj, schema=slot)
        return self.patch_object(obj=db_obj, schema=slot)
