from typing import List, Optional
from pydantic import root_validator

from .base import (
    BaseSchema, optional,
    slug_type, name_type,
)
from .role import RoleSchema
from .scope import ScopeSchema
from .staff import StaffSchema
from .schedule import ScheduleCompactSchema
from watcher.logic.exceptions import InvalidCompositionInput


class CompositionBaseSchema(BaseSchema):
    name: name_type
    full_service: bool
    autoupdate: bool


class CompositionSchemaWithRefsIds(BaseSchema):
    roles: List[int]
    scopes: List[int]
    staff: List[int]

    excluded_roles: List[int]
    excluded_scopes: List[int]
    excluded_staff: List[int]


class CompositionSchemaWithRefsObj(BaseSchema):
    roles: List[RoleSchema]
    scopes: List[ScopeSchema]
    staff: List[StaffSchema]

    excluded_roles: List[RoleSchema]
    excluded_scopes: List[ScopeSchema]
    excluded_staff: List[StaffSchema]


@optional(
    'full_service',
    'autoupdate',
    'roles',
    'scopes',
    'staff',
    'excluded_roles',
    'excluded_scopes',
    'excluded_staff',
)
class CompositionCreateSchema(CompositionBaseSchema, CompositionSchemaWithRefsIds):
    slug: slug_type
    service_id: int

    @root_validator
    def check_expected_fields(cls, values: dict) -> dict:
        one_of_expected = (
            'full_service', 'roles',
            'scopes', 'staff',
        )
        for field in one_of_expected:
            if values.get(field):
                break
        else:
            raise InvalidCompositionInput

        return values


@optional
class CompositionPatchSchema(CompositionBaseSchema, CompositionSchemaWithRefsIds):
    pass


class CompositionListSchema(CompositionBaseSchema):
    id: int
    slug: slug_type
    service_id: int
    participants: List[StaffSchema]


class CompositionGetSchema(CompositionListSchema, CompositionSchemaWithRefsObj):
    participants: List[StaffSchema]
    schedules: Optional[List[ScheduleCompactSchema]]
