import datetime

from typing import Optional, List
from pydantic import validator, root_validator, constr

from watcher.enums import ScheduleState, IntervalRotation
from watcher.config import settings
from watcher.logic.exceptions import ScheduleMaxDays
from .base import (
    BaseSchema, optional,
    name_type, slug_type,
)
from .service import ServiceSchema
from .problem import ProblemBaseSchema
from .revision import RevisionBaseSchema


class ScheduleBaseSchema(BaseSchema):
    name: name_type
    description: Optional[constr(max_length=5000)]
    state: ScheduleState
    schedules_group_id: Optional[int]
    days_for_notify_of_problems: datetime.timedelta
    days_for_notify_of_begin: List[datetime.timedelta]
    length_of_absences: datetime.timedelta
    pin_shifts: datetime.timedelta
    recalculate: bool
    threshold_day: datetime.timedelta
    days_before_vacation: datetime.timedelta
    days_after_vacation: datetime.timedelta
    rotation: IntervalRotation


class ScheduleWithValidatorSchema(ScheduleBaseSchema):
    show_in_services: Optional[set[int]]

    @validator('days_for_notify_of_problems')
    def check_days_for_notify_of_problems(cls, v):
        if v and v.total_seconds() % 86400:
            raise ValueError('value should be full day')
        if v and v > datetime.timedelta(days=settings.MAX_PROBLEM_OR_NOTIFICATION_DAYS):
            raise ScheduleMaxDays
        return v

    @root_validator(pre=True)
    def check_if_days_for_notify_supplied(cls, values):
        # Если в JSON не было значений, то выставляем дефолтные
        # Явно переданный None трактуется как отключение уведомлений
        if 'days_for_notify_of_problems' not in values:
            values['days_for_notify_of_problems'] = datetime.timedelta(days=14)
        if 'days_for_notify_of_begin' not in values:
            values['days_for_notify_of_begin'] = [
                datetime.timedelta(days=0),
                datetime.timedelta(days=1),
                datetime.timedelta(days=7)
            ]
        return values

    @validator('days_for_notify_of_begin')
    def check_days_for_notify_of_begin(cls, v):
        if v:
            for interval in v:
                if interval.total_seconds() % 86400:
                    raise ValueError('value should be full day')
                if interval > datetime.timedelta(days=settings.MAX_PROBLEM_OR_NOTIFICATION_DAYS):
                    raise ScheduleMaxDays
        return v

    @validator('threshold_day')
    def check_threshold_day(cls, v):
        if v and v.total_seconds() > settings.MAX_THRESHOLD_DAYS:
            raise ScheduleMaxDays(message={
                'ru': f'Максимально допустимое значение - {settings.MAX_THRESHOLD_DAYS} дней',
                'en': f'The maximum allowed value is {settings.MAX_THRESHOLD_DAYS} days'
            })
        if v and v.total_seconds() < settings.MIN_THRESHOLD_DAYS:
            raise ScheduleMaxDays(message={
                'ru': f'Минимально допустимое значение - {settings.MIN_THRESHOLD_DAYS} дней',
                'en': f'The minimum allowed value is {settings.MIN_THRESHOLD_DAYS} days'
            })

        if v and v.total_seconds() % 86400:
            raise ValueError('value should be full day')
        return v

    @validator('days_before_vacation', 'days_after_vacation')
    def check_days_vacation(cls, v):
        if v and v.total_seconds() % 86400:
            raise ValueError('value should be full day')

        if v and v > datetime.timedelta(days=settings.MAX_PROBLEM_OR_NOTIFICATION_DAYS):
            raise ScheduleMaxDays(message={
                'ru': f'Максимально допустимое значение - {settings.MAX_DAYS_BEFORE_OR_AFTER_VACATION} дней',
                'en': f'The maximum allowed value is {settings.MAX_DAYS_BEFORE_OR_AFTER_VACATION} days'
            })
        return v


@optional(
    'state',
    'days_for_notify_of_problems',
    'days_for_notify_of_begin',
    'length_of_absences',
    'pin_shifts',
    'recalculate',
    'threshold_day',
    'show_in_services',
    'rotation',
)
class ScheduleCreateSchema(ScheduleWithValidatorSchema):
    slug: slug_type
    service_id: int
    start: Optional[datetime.datetime]
    days_for_notify_of_problems: Optional[datetime.timedelta]
    days_for_notify_of_begin: Optional[List[datetime.timedelta]]
    days_before_vacation: Optional[datetime.timedelta]
    days_after_vacation: Optional[datetime.timedelta]
    rotation: Optional[IntervalRotation]


class ScheduleListBaseSchema(ScheduleBaseSchema):
    id: int
    slug: slug_type
    service_id: int
    author_id: int
    days_for_notify_of_problems: Optional[datetime.timedelta]
    days_for_notify_of_begin: Optional[List[datetime.timedelta]]
    recalculation_in_process: bool
    revision: Optional[RevisionBaseSchema]
    schedules_group: 'ScheduleGroupCompactSchema'
    service: ServiceSchema
    is_responsible: Optional[bool]
    is_important: bool
    rotation: Optional[IntervalRotation]


class ScheduleListSchema(ScheduleListBaseSchema):
    actual_problems: List[ProblemBaseSchema]
    show_in_services: List[ServiceSchema]
    show_in_staff: Optional[bool]


class ScheduleRichListSchema(ScheduleListBaseSchema):
    service: ServiceSchema


class ScheduleCompactSchema(BaseSchema):
    id: int
    name: name_type
    slug: slug_type
    service_id: int
    author_id: int
    is_responsible: Optional[bool]


class ScheduleABCSchema(BaseSchema):
    id: int
    name: name_type
    slug: slug_type
    service_id: int


# workaround of circular import
# https://stackoverflow.com/a/63425646/4227169
from watcher.api.schemas.schedulegroup import ScheduleGroupCompactSchema
ScheduleListSchema.update_forward_refs()
ScheduleRichListSchema.update_forward_refs()


@optional
class SchedulePatchSchema(ScheduleWithValidatorSchema):
    days_for_notify_of_problems: Optional[datetime.timedelta]
