import logging
from logging.config import dictConfig

from fastapi import FastAPI
from fastapi.middleware.gzip import GZipMiddleware
from fastapi.routing import APIRoute
from fastapi.exceptions import RequestValidationError

from asgi_yauth.middleware import (
    YauthTestMiddleware,
    YauthMiddleware,
)

from watcher.logic.exceptions import BaseWatcherException
from watcher.logic.exception_handler import (
    base_exception_handler,
    unhandled_exception_handler,
    request_validation_exception_handler_watcher,
)
from .middleware import (
    yauth_require,
    set_tvm_user,
    log_context,
    x_secret_token,
)

from .logging import LOGGING_CONFIG
from .config import settings
from .router import api_router, frontend_router
from .common_routes import router as common_router


dictConfig(LOGGING_CONFIG)

logger = logging.getLogger(__name__)


def generate_operation_ids(app: FastAPI) -> None:
    for route in app.routes:
        if isinstance(route, APIRoute):
            route.operation_id = route.name


def add_middlewares(app: FastAPI) -> None:
    """
    Тут важен порядок, middleware добавленые
    в начале этого метода будут вызваны после тех
    что идут в конце
    """
    app.add_middleware(GZipMiddleware, minimum_size=1000)
    app.middleware('http')(log_context)
    app.middleware('http')(set_tvm_user)
    app.middleware('http')(yauth_require)

    if settings.ENV_TYPE != settings.ENV_TYPE_PRODUCTION:
        app.middleware('http')(x_secret_token)
    if settings.ENV_TYPE in settings.ENV_TYPE_FOR_UNIT_TEST:
        yauth_middleware = YauthTestMiddleware
    else:
        yauth_middleware = YauthMiddleware
    app.add_middleware(yauth_middleware, config=settings)


def bind_exception_handlers(app: FastAPI) -> None:
    app.exception_handler(BaseWatcherException)(base_exception_handler)
    app.exception_handler(Exception)(unhandled_exception_handler)
    app.exception_handler(RequestValidationError)(request_validation_exception_handler_watcher)


def get_app():
    app = FastAPI(
        title='watcher',
        openapi_url=f'{settings.DOCS_PREFIX}/openapi.json',
        docs_url=f'{settings.DOCS_PREFIX}/swagger',
        redoc_url=f'{settings.DOCS_PREFIX}/redoc',
    )
    app.include_router(api_router, prefix='/api/watcher/v1')
    app.include_router(frontend_router, prefix='/api/watcher/frontend')
    app.include_router(common_router, prefix='/api/watcher/common')
    generate_operation_ids(app)

    add_middlewares(app)
    bind_exception_handlers(app)
    return app
