from telegram import Update, Bot, ParseMode
from telegram.utils.request import Request
from telegram.ext import (
    Updater,
    CallbackContext,
    CommandHandler,
)
from sqlalchemy.orm import Session

from watcher.config import settings
from watcher.logic.tvm_client import get_tvm2_ticket
from watcher.logic.clients.staff_api import staff_api_client
from watcher.logic.clients.jns import jns_client
from watcher.db.base import dbconnect
from watcher.crud.bot_user import (
    get_bot_user_by_id,
    create_bot_user,
    get_or_create_chat,
)
from watcher.enums import ChatType


class StackRequest(Request):
    def _request_wrapper(self, method, url, *args, **kwargs):
        headers = kwargs.get('headers') or {}
        headers['x-ya-service-ticket'] = get_tvm2_ticket('gozora')
        headers['x-ya-client-id'] = settings.GOZORA_CLIENT_ID
        headers['x-ya-dest-url'] = url

        kwargs['headers'] = headers

        return super()._request_wrapper(
            method, settings.GOZORA_URL,
            *args, **kwargs
        )


class StackBot(Bot):
    pass


def get_bot(token: str) -> Bot:
    if settings.BOT_USE_GOZORA:
        request = StackRequest(
            con_pool_size=settings.BOT_WORKERS + 4
        )
        bot = StackBot(token, request=request)
    else:
        request = Request(con_pool_size=5)
        bot = Bot(token, request=request)
    return bot


class BotHandler:
    def __init__(self) -> None:
        kwargs = {
            'workers': settings.BOT_WORKERS,
        }
        if settings.BOT_USE_GOZORA:
            kwargs['bot'] = get_bot(settings.BOT_TOKEN)
        else:
            kwargs['token'] = settings.BOT_TOKEN

        self.updater = Updater(**kwargs)
        self._add_handlers()

    def _add_handlers(self) -> None:
        dispatcher = self.updater.dispatcher

        dispatcher.add_handler(CommandHandler(
            command='start',
            callback=self._start
        ))

    def _get_bot_id(self):
        return settings.BOT_TOKEN.split(':')[0]

    def run(self) -> None:
        if settings.BOT_USE_LOCKS:
            """
            тут нужно брать лок если его нет, а если есть - спать 5 минут
            а потом снова пробовать взять

            не забыть что лок еще и подновлять нужно

            ну или через базу локи сделать
            """
        else:
            self.updater.start_polling()
        self.updater.idle()

    @dbconnect
    def _start(self, update: Update, context: CallbackContext, session: Session) -> None:
        if update.effective_chat.type in ChatType.group_types():
            message = 'Group chats not supported'
        else:
            bot_user = get_bot_user_by_id(db=session, telegram_id=update.effective_user.id)
            if bot_user:
                message = f'Already authorized: `@{bot_user.staff.login}`'
            else:
                staff = staff_api_client.get_by_telegram(username=update.effective_user.username)

                if staff:
                    jns_client.upsert_user(login=staff.login, chat_id=update.effective_chat.id)
                    bot_user = create_bot_user(
                        db=session, staff_uid=staff.uid,
                        username=update.effective_user.username,
                        telegram_id=update.effective_user.id,
                    )
                    get_or_create_chat(
                        db=session,
                        chat_data=update.effective_chat,
                        author_id=bot_user.id,
                    )
                    message = f'Successfull authorization: `@{staff.login}`'
                else:
                    message = 'Unknown account'
        context.bot.send_message(
            chat_id=update.effective_chat.id,
            text=message,
            parse_mode=ParseMode.MARKDOWN,
        )
