from typing import Optional, Iterable, Union

from sqlalchemy import or_
from sqlalchemy.orm import Session, joinedload
from sqlalchemy.orm.query import Query
from sqlalchemy.exc import IntegrityError
from sqlalchemy.sql.expression import insert

from watcher.db import BaseModel
from watcher.logic.exceptions import RecordNotFound
from watcher.logic.filter import prepare_filter_params, extract_joined_params
from watcher.logic.exceptions import BadRequest, WatcherAttributeError
from watcher.api.schemas.base import BaseSchema


def _add_joined_load(query: Query, joined_load: Optional[tuple] = None) -> Query:
    if joined_load:
        query = query.options(
            *[
                joinedload(table)
                for table in joined_load
            ]
        )
    return query


def get_object_by_model(
    db: Session, model: BaseModel, object_id: Union[int, str],
    joined_load: Optional[tuple] = None,
    field: Optional[str] = None
) -> Optional[BaseModel]:
    query = db.query(model)
    query = _add_joined_load(query=query, joined_load=joined_load)
    if field:
        return query.filter(getattr(model, field)==object_id).first()
    return query.get(object_id)


def get_object_by_model_or_404(db: Session, model: BaseModel, object_id: int, joined_load: Optional[tuple] = None) -> BaseModel:
    object = get_object_by_model(db=db, model=model, object_id=object_id, joined_load=joined_load)
    if not object:
        raise RecordNotFound()
    return object


def list_objects_by_model(db: Session, model: BaseModel, joined_load: Optional[tuple] = None) -> Query:
    query = db.query(model)
    return _add_joined_load(query=query, joined_load=joined_load)


def query_objects_by_ids(
    db: Session, model: BaseModel, object_ids: Iterable[int],
    joined_load: Optional[tuple] = None,
) -> Query:
    query = db.query(model).filter(getattr(model, 'id').in_(object_ids))
    return _add_joined_load(query=query, joined_load=joined_load)


def filter_joined_objects(query, table, table_filter_params):
    """
    Добавляет необходимые джойны к запросу, возвращает запрос с джойнами и подготовленные фильтры
    например, для того чтобы отфильтровать запрос к Shift по полю 'service.owner.login' ==
    нужно написать следующий код
    query.join(Shift.service, isouter=True).join(Service.owner, isouter=True).filter(Staff.login == ...)

    query - изначальный запрос
    table - ожидается строка с именем присоединяемой таблицы вида 'service.owner'
    table_filter_params - ожидается словарь для фильтрации вида  {'login': 'sa', ...}
    """
    primary_model = query.column_descriptions[0]['expr']
    model = primary_model
    for joined_model_name in table.split('.'):
        try:
            field = getattr(model, joined_model_name)
        except AttributeError:
            raise WatcherAttributeError(obj_name=model.__name__, attr_name=joined_model_name)
        query = query.join(field, isouter=True)
        model = field.property.mapper.class_
    filter_params = prepare_filter_params(
        filter_params=table_filter_params,
        model=model,
    )
    return query, filter_params


def filter_objects(query: Query, filter_params: dict, or_statement: bool = False) -> Query:
    """
    filter_params - ожидается словарь для фильтрации вида
    {'id': 1, 'slot_id__in': [1,2,3], 'value__gt': 40, ...}
    or_statement - переданные условия в словаре будут применениы через логическое 'ИЛИ'
    """
    if not filter_params:
        return query
    for or_key in list(filter_params):
        if or_key.startswith('|'):
            query = filter_objects(
                query=query, filter_params=filter_params.pop(or_key),
                or_statement=True,
            )
    filter_params_list = []
    joined_params = extract_joined_params(filter_params)
    for table, table_filter_params in joined_params.items():
        query, joined_filter_params = filter_joined_objects(
            query=query, table=table,
            table_filter_params=table_filter_params,
        )
        filter_params_list.extend(joined_filter_params)
    model = query.column_descriptions[0]['expr']
    filter_params = prepare_filter_params(
        filter_params=filter_params,
        model=model,
    )
    filter_params_list.extend(filter_params)
    if or_statement:
        return query.filter(or_(*filter_params_list))
    else:
        return query.filter(*filter_params_list)


# TODO: это не коммит оджного объект, это коммит всего что добавлено
#   нужно переделать, например, сделать через дочернюю сессию
def commit_object(db: Session, obj: BaseModel) -> BaseModel:
    try:
        db.add(obj)
        db.commit()
        db.refresh(obj)
    except IntegrityError as exc:
        db.rollback()
        raise BadRequest(error=repr(exc))
    return obj


def _set_fields(obj: BaseModel, update_data: dict) -> bool:
    has_changes = False
    for field, value in update_data.items():
        if not hasattr(obj, field):
            continue

        current_value = getattr(obj, field)
        if current_value != value:
            has_changes = True
            setattr(obj, field, value)

    return has_changes


def patch_object(
    db: Session,
    obj: BaseModel,
    schema: BaseSchema,
    commit: bool = True,
) -> BaseModel:
    update_data = schema.dict(exclude_unset=True)
    update_data.pop('id', None)
    has_changes = _set_fields(obj=obj, update_data=update_data)

    if has_changes and commit:
        db.commit()

    return obj


def update_many_to_many_for_field(
    db: Session, obj: BaseModel, target: set, current: set,
    table: BaseModel, field_key: str, current_refs: Query,
    related_field: str,
) -> None:
    """
    Обновляет many-to-many связи - получает на вход список текущих и нужных
    id сущностей, а так же набор вспомогательных данных

    Добавляет новые связи, удаляет пропавшие, оставляет как есть если изменения
    не нужны

    :param current_refs - текущие записи из прокси модели
    :param related_field - поле в прокси модели, например
    для CompositionParticipants это composition_id

    :param field_key - поле в прокси модели отвечающее за связь с целевой
    моделью  например для CompositionParticipants это staff_id
    """
    to_add = target.difference(current)
    to_delete = current.difference(target)

    if to_add:
        _bulk_insert_refs(
            db=db, obj=obj,
            field_key=field_key,
            to_add=to_add, table=table,
            related_field=related_field,
        )
    if to_delete:
        _bulk_delete_refs(
            db=db, table=table, current_refs=current_refs,
            field_key=field_key, to_delete=to_delete,
        )


def _bulk_delete_refs(
    db: Session, table: BaseModel, current_refs: Iterable,
    field_key: str, to_delete: set
) -> None:
    """
    Получает на вход список текущих прокси записей в many-to-many
    пробегается по ним и выбирает те у которых id связанный модели
    находится в списке на удаление - и удаляет эти связи
    """
    ids_to_delete = [
        ref.id for ref in current_refs
        if getattr(ref, field_key) in to_delete
    ]
    db.query(table).filter(table.id.in_(ids_to_delete)).delete(synchronize_session=False)


def _bulk_insert_refs(
    db: Session, obj: BaseModel, to_add: Iterable,
    table: BaseModel, field_key: str, related_field: str
) -> None:
    """
    Балком добавляет связи в прокси модель.
    Нужно для сокращение запросов при создании many-to-many связей
    """
    values_to_insert = [
        {related_field: obj.id, field_key: value}
        for value in to_add
    ]
    if values_to_insert:
        db.execute(insert(table).values(values_to_insert))


def _remove_relationships(dict_obj: dict, relationship_fields: Iterable) -> dict:
    """
    Убирает из словаря схемы переданные объекты, используется для
    выборки many-to-many связей для последующей их обработки
    """
    many_to_many = {}
    for field in relationship_fields:
        value = dict_obj.pop(field, None)
        if value is not None:
            many_to_many[field] = value

    return many_to_many
