from typing import Optional

from telegram.chat import Chat as TelegramChat
from sqlalchemy.orm import Session

from watcher.crud.base import get_object_by_model
from watcher.db import BotUser, Chat
from watcher.enums import ChatType
from .staff import get_staff_by_uid
from .base import commit_object


def get_bot_user_by_id(db: Session, telegram_id: int) -> Optional[BotUser]:
    return get_object_by_model(
        db=db, model=BotUser, object_id=str(telegram_id),
        field='telegram_id',
        joined_load=('staff', )
    )


def query_bot_user_by_staff_ids(db: Session, staff_ids: list[int]) -> list[BotUser]:
    return db.query(BotUser).filter(
        BotUser.staff_id.in_(staff_ids)
    ).all()


def create_bot_user(
    db: Session, staff_uid: str,
    telegram_id: int, username: str,
) -> BotUser:
    staff = get_staff_by_uid(db=db, uid=staff_uid)
    bot_user = BotUser(
        staff_id=staff.id,
        telegram_id=telegram_id,
        username=username,
    )
    return commit_object(db=db, obj=bot_user)


def get_or_create_chat(db: Session, chat_data: TelegramChat, author_id: int) -> Chat:
    chat = db.query(Chat).get(chat_data.id)
    if not chat:
        if chat_data.type == ChatType.group:
            title = chat_data.title
        else:
            title = chat_data.username
        chat = Chat(
            id=chat_data.id,
            author_id=author_id,
            title=title,
            chat_type=chat_data.type,
        )
        db.add(chat)
        db.commit()
        db.refresh(chat)
    return chat
