import logging
from functools import lru_cache, wraps
from typing import Iterator

from sqlalchemy import create_engine
from sqlalchemy.orm import Session, declarative_base
from sqlalchemy import (
    Column,
    types,
)
from fastapi_utils.session import FastAPISessionMaker

from watcher.config import settings
from watcher.logic.timezone import now

logger = logging.getLogger(__name__)


def get_db() -> Iterator[Session]:
    """ FastAPI dependency that provides a sqlalchemy session """
    yield from _get_fastapi_sessionmaker().get_db()


@lru_cache()
def _get_fastapi_sessionmaker() -> FastAPISessionMaker:
    database_url = settings.database_url
    return FastAPISessionMaker(database_url)


BaseModel = declarative_base()


class TimestampedModelMixin:
    created_at = Column(
        types.DateTime(timezone=True),
        default=now,
    )
    updated_at = Column(
        types.DateTime(timezone=True),
        default=now,
        onupdate=now,
    )


def dbconnect(func):
    @wraps(func)
    def wrapper(*args, **kwargs):
        session = kwargs.get('session')
        if not session:
            engine = create_engine(settings.database_url)
            session = Session(bind=engine, )
            kwargs['session'] = session
        try:
            func(*args, **kwargs)
            session.commit()
        except Exception as exc:
            logger.exception(f'Unexpected error in {func.__name__}: {repr(exc)}')
            session.rollback()
            raise
        finally:
            session.close()
    return wrapper
