from sqlalchemy import (
    Column,
    ForeignKey,
    types,
    UniqueConstraint,
)
from sqlalchemy.orm import relationship

from .base import BaseModel, TimestampedModelMixin
from watcher import enums


class Composition(TimestampedModelMixin, BaseModel):
    """
    Описание настроек состава
    """
    __tablename__ = 'composition'

    id = Column(types.Integer, primary_key=True, index=True)

    slug = Column(types.String, nullable=False)
    name = Column(types.String, nullable=False)

    # непосредственно настройки
    composition_type = Column(
        types.Enum(enums.CompositionType),
        nullable=False,
        default=enums.CompositionType.role,
    )

    roles = relationship('Role', secondary='composition_roles', backref='used_in_compositions', )
    scopes = relationship('Scope', secondary='composition_scopes', backref='used_in_compositions', )
    staff = relationship('Staff', secondary='composition_staff', backref='used_in_compositions', )

    excluded_roles = relationship('Role', secondary='composition_excluded_roles', backref='excluded_from_compositions', )
    excluded_scopes = relationship('Scope', secondary='composition_excluded_scopes', backref='excluded_from_compositions', )
    excluded_staff = relationship('Staff', secondary='composition_excluded_staff', backref='excluded_from_compositions', )

    participants = relationship('Staff', secondary='composition_participants', backref='compositions')

    full_service = Column(
        types.Boolean,
        default=False,
        info={
            'verbose_name': 'Признак, что в составе участвует весь сервис'
        }
    )

    service_id = Column(
        types.Integer,
        ForeignKey('services_service.id'),
        nullable=False,
        info={
            'verbose_name': 'Сервис, к которому привязан состав'
        }
    )
    service = relationship('Service', foreign_keys=[service_id])

    autoupdate = Column(
        types.Boolean,
        default=True,
        info={
            'verbose_name': 'Обновлять состав автоматически при добавлении в роль/скоуп/сервис новых участников'
        }
    )

    # технические поля
    __table_args__ = (
        UniqueConstraint('service_id', 'slug', name='uniq_service_composition'),
    )


class CompositionParticipants(TimestampedModelMixin, BaseModel):
    """
    Состав участников
    """
    __tablename__ = 'composition_participants'

    id = Column(types.Integer, primary_key=True, index=True)
    composition_id = Column(types.Integer, ForeignKey('composition.id', ondelete='CASCADE'), nullable=False,)
    composition = relationship('Composition', overlaps='participants,compositions')
    staff_id = Column(types.Integer, ForeignKey('intranet_staff.id'), nullable=False,)
    staff = relationship('Staff', overlaps='participants,compositions')


class CompositionToRole(TimestampedModelMixin, BaseModel):
    """
    many-to-many композиция - роли
    """
    __tablename__ = 'composition_roles'

    id = Column(types.Integer, primary_key=True, index=True)
    composition_id = Column(types.Integer, ForeignKey('composition.id', ondelete='CASCADE'), nullable=False,)
    composition = relationship('Composition', overlaps='roles,used_in_compositions')
    role_id = Column(types.Integer, ForeignKey('roles_role.id'), nullable=False,)
    role = relationship('Role', overlaps='roles,used_in_compositions')


class CompositionToScope(TimestampedModelMixin, BaseModel):
    """
    many-to-many композиция - scope
    """
    __tablename__ = 'composition_scopes'

    id = Column(types.Integer, primary_key=True, index=True)
    composition_id = Column(types.Integer, ForeignKey('composition.id', ondelete='CASCADE'), nullable=False,)
    composition = relationship('Composition', overlaps='scopes,used_in_compositions')
    scope_id = Column(types.Integer, ForeignKey('roles_rolescope.id'), nullable=False,)
    scope = relationship('Scope', overlaps='scopes,used_in_compositions')


class CompositionToStaff(TimestampedModelMixin, BaseModel):
    """
    many-to-many композиция - staff
    это именно для настроек, это не состав конечный
    """
    __tablename__ = 'composition_staff'

    id = Column(types.Integer, primary_key=True, index=True)
    composition_id = Column(types.Integer, ForeignKey('composition.id', ondelete='CASCADE'), nullable=False,)
    composition = relationship('Composition', overlaps='staff,used_in_compositions')
    staff_id = Column(types.Integer, ForeignKey('intranet_staff.id'), nullable=False,)
    staff = relationship('Staff', overlaps='staff,used_in_compositions')


class CompositionToRoleExcluded(TimestampedModelMixin, BaseModel):
    """
    исключения many-to-many композиция - роли
    """
    __tablename__ = 'composition_excluded_roles'

    id = Column(types.Integer, primary_key=True, index=True)
    composition_id = Column(types.Integer, ForeignKey('composition.id', ondelete='CASCADE'), nullable=False,)
    composition = relationship('Composition', overlaps='excluded_roles,excluded_from_compositions')
    role_id = Column(types.Integer, ForeignKey('roles_role.id'), nullable=False,)
    role = relationship('Role', overlaps='excluded_roles,excluded_from_compositions')


class CompositionToScopeExcluded(TimestampedModelMixin, BaseModel):
    """
    исключения many-to-many композиция - scope
    """
    __tablename__ = 'composition_excluded_scopes'

    id = Column(types.Integer, primary_key=True, index=True)
    composition_id = Column(types.Integer, ForeignKey('composition.id', ondelete='CASCADE'), nullable=False,)
    scope_id = Column(types.Integer, ForeignKey('roles_rolescope.id'), nullable=False,)


class CompositionToStaffExcluded(TimestampedModelMixin, BaseModel):
    """
    исключения many-to-many композиция - staff
    это именно для настроек, это не состав конечный
    """
    __tablename__ = 'composition_excluded_staff'

    id = Column(types.Integer, primary_key=True, index=True)
    composition_id = Column(types.Integer, ForeignKey('composition.id', ondelete='CASCADE'), nullable=False,)
    composition = relationship('Composition', overlaps='excluded_staff,excluded_from_compositions')
    staff_id = Column(types.Integer, ForeignKey('intranet_staff.id'), nullable=False,)
    staff = relationship('Staff', overlaps='excluded_staff,excluded_from_compositions')
