from sqlalchemy import Column, types, ForeignKey
from sqlalchemy.orm import relationship

from .base import BaseModel
from watcher import enums


class Staff(BaseModel):
    __tablename__ = 'intranet_staff'

    id = Column(types.Integer, primary_key=True, index=True)
    uid = Column(types.Integer, index=True, unique=True)

    chief_id = Column(types.Integer, ForeignKey('intranet_staff.id'))
    chief = relationship('Staff', remote_side=id,)

    user_id = Column(types.Integer, index=True)
    login = Column(types.String, index=True, default='')
    first_name = Column(types.String, index=True)
    first_name_en = Column(types.String, index=True)
    last_name = Column(types.String, index=True)
    last_name_en = Column(types.String, index=True)
    is_robot = Column(types.Boolean, default=True)
    affiliation = Column(types.Enum(enums.StaffAffilation), index=True)
    gender = Column(types.Enum(enums.StaffGender), index=True)
    is_dismissed = Column(types.Boolean, default=False)
    join_at = Column(types.DateTime)
    quit_at = Column(types.DateTime)
    lang_ui = Column(types.Enum(enums.StaffLang), index=True)
    work_email = Column(types.String, index=True, default='')
    staff_id = Column(types.Integer, index=True)
    telegram_account = Column(types.String)
    created_at = Column(types.DateTime)
    modified_at = Column(types.DateTime)


class Service(BaseModel):
    __tablename__ = 'services_service'

    id = Column(types.Integer, primary_key=True, index=True)

    parent_id = Column(types.Integer, ForeignKey('services_service.id'))
    parent = relationship('Service', remote_side=id, backref='childrens')
    owner_id = Column(types.Integer, ForeignKey('intranet_staff.id'), nullable=True)
    owner = relationship('Staff')

    level = Column(types.Integer, index=True)
    state = Column(
        types.Enum(enums.ServiceState),
        index=True,
        default=enums.ServiceState.develop,
    )
    name = Column(types.String)
    name_en = Column(types.String)
    slug = Column(types.String, unique=True, nullable=False)
    description = Column(types.String)
    description_en = Column(types.String)
    is_base = Column(types.Boolean, default=False)
    readonly_state = Column(
        types.Enum(enums.ServiceReadonlyStatus),
        nullable=True, default=None
    )
    readonly_start_time = Column(types.DateTime, nullable=True)
    staff_id = Column(types.Integer, nullable=True, index=True)
    path = Column(types.String, nullable=True, default='', info={'verbose_name': 'Путь по slug проектов'})
    ancestors = Column(types.JSON, default=dict, nullable=True)
    is_exportable = Column(types.Boolean, default=False, info={'verbose_name': 'Доступен в апи по умолчанию'})
    created_at = Column(types.DateTime)
    modified_at = Column(types.DateTime)


class Role(BaseModel):
    __tablename__ = 'roles_role'

    id = Column(types.Integer, primary_key=True, index=True)

    EXCLUSIVE_OWNER = 'product_head'
    DEPUTY_OWNER = 'product_deputy_head'
    RESPONSIBLE = 'responsible'
    DUTY = 'duty'
    RESPONSIBLE_FOR_DUTY = 'responsible_for_duty'
    SERVICE_FINANCIAL_RESOURCES_REQUESTER = 'financial_resources_requester'
    HEADS = {
        EXCLUSIVE_OWNER,
        DEPUTY_OWNER,
    }
    RESPONSIBLES = HEADS | {RESPONSIBLE}
    CAN_NOT_USE_FOR_DUTY = RESPONSIBLES
    EXCLUDE_FROM_DUTY = CAN_NOT_USE_FOR_DUTY | {RESPONSIBLE_FOR_DUTY}

    WATCHER_RESPONSIBLES = [
        RESPONSIBLE_FOR_DUTY, RESPONSIBLE,
        DEPUTY_OWNER, EXCLUSIVE_OWNER,
    ]  # тут порядок имеет значение при отправке уведомлений

    name = Column(types.String, index=True)
    name_en = Column(types.String, index=True)
    code = Column(types.String, default='other', index=True)
    created_at = Column(types.DateTime)
    modified_at = Column(types.DateTime)
    is_exportable = Column(types.Boolean, default=True, info={'verbose_name': 'Роль которая экспортируется в стафф'})

    projected_role_id = Column(types.Integer, ForeignKey('roles_role.id'))
    projected_role = relationship('Role', remote_side=id,)
    service_id = Column(types.Integer, ForeignKey('services_service.id'), nullable=True)
    service = relationship('Service')
    scope_id = Column(types.Integer, ForeignKey('roles_rolescope.id'), nullable=True)
    scope = relationship('Scope', backref='roles')


class Scope(BaseModel):
    __tablename__ = 'roles_rolescope'

    id = Column(types.Integer, primary_key=True, index=True)
    slug = Column(types.String, unique=True)
    name = Column(types.String)
    name_en = Column(types.String)
    protected = Column(types.Boolean)
    can_issue_at_duty_time = Column(types.Boolean)


class Member(BaseModel):
    __tablename__ = 'services_servicemember'

    id = Column(types.Integer, primary_key=True, index=True)

    staff_id = Column(types.Integer, ForeignKey('intranet_staff.id'))
    staff = relationship('Staff')
    service_id = Column(types.Integer, ForeignKey('services_service.id'))
    service = relationship('Service')
    role_id = Column(types.Integer, ForeignKey('roles_role.id'))
    role = relationship('Role')
    from_department_id = Column(types.Integer, nullable=True)

    state = Column(
        types.Enum(enums.MemberState),
        nullable=False,
        default=enums.MemberState.requested,
    )
    autorequested = Column(types.Boolean, default=False)
    custom_role = Column(types.String, nullable=True, info={'verbose_name': 'Своё название роли'})
    position = Column(types.Integer, index=True, default=0)
    is_temp = Column(types.Boolean, default=False, info={'verbose_name': 'Временно исполняющий обязанности'})
    created_at = Column(types.DateTime)
    modified_at = Column(types.DateTime)
    found_in_staff_at = Column(types.DateTime, index=True, nullable=True)


class Gap(BaseModel):
    __tablename__ = 'duty_gap'

    id = Column(types.Integer, primary_key=True, index=True)

    staff_id = Column(types.Integer, ForeignKey('intranet_staff.id'))
    staff = relationship('Staff')

    status = Column(
        types.Enum(enums.GapStatus),
        nullable=False,
        default=enums.GapStatus.active,
        info={
            'verbose_name': 'Статус отсутствия'
        },
    )
    work_in_absence = Column(types.Boolean, info={'verbose_name': 'Будет работать во время отсутствия'})
    full_day = Column(types.Boolean, info={'verbose_name': 'Отсутствие на весь день'})
    type = Column(types.String, info={'verbose_name': 'Тип отсутствия'})
    created_at = Column(types.DateTime(timezone=True), nullable=False)
    deleted_at = Column(types.DateTime(timezone=True))
    start = Column(types.DateTime(timezone=True))
    end = Column(types.DateTime(timezone=True))
    gap_id = Column(types.BigInteger, unique=True, index=True, info={'verbose_name': 'Id в Гэпе'})

    def __lt__(self, other):
        return self.end < other.end


class Holiday(BaseModel):
    __tablename__ = "holidays_holiday"

    date = Column(types.Date, primary_key=True)


class UserPermission(BaseModel):
    __tablename__ = "auth_user_user_permissions"

    id = Column(types.Integer, primary_key=True, index=True)
    user_id = Column(types.Integer, index=True)
    permission_id = Column(types.Integer, index=True)
