from .base import BaseModel, TimestampedModelMixin
from sqlalchemy import (
    Column,
    ForeignKey,
    types,
)
from sqlalchemy.ext.associationproxy import association_proxy
from sqlalchemy.orm import relationship, backref

from watcher import enums


class ManualGapSettings(TimestampedModelMixin, BaseModel):
    __tablename__ = 'manual_gap_settings'

    id = Column(types.Integer, primary_key=True)
    title = Column(types.String, nullable=False)
    comment = Column(types.String)

    staff_id = Column(types.Integer, ForeignKey('intranet_staff.id', ondelete='CASCADE'), nullable=False)
    staff = relationship('Staff', foreign_keys=[staff_id], backref='manual_gap_settings')

    start = Column(types.DateTime(timezone=True), nullable=False,)
    end = Column(types.DateTime(timezone=True), nullable=False,)

    is_active = Column(types.Boolean, default=True)

    recurrence = Column(types.Enum(enums.ManualGapRecurrence), default=enums.ManualGapRecurrence.once)

    all_services = Column(types.Boolean, default=False)
    services = relationship('Service', secondary='manual_gap_settings_services')
    schedules = relationship('Schedule', secondary='manual_gap_settings_schedules')


class ManualGapSettingsServices(TimestampedModelMixin, BaseModel):
    __tablename__ = 'manual_gap_settings_services'

    id = Column(types.Integer, primary_key=True)

    service_id = Column(types.Integer, ForeignKey('services_service.id', ondelete='CASCADE'), )
    service = relationship('Service', viewonly=True)

    gap_settings_id = Column(types.Integer, ForeignKey('manual_gap_settings.id', ondelete='CASCADE'), )
    gap_settings = relationship('ManualGapSettings', viewonly=True)


class ManualGapSettingsSchedules(TimestampedModelMixin, BaseModel):
    __tablename__ = 'manual_gap_settings_schedules'

    id = Column(types.Integer, primary_key=True)

    schedule_id = Column(types.Integer, ForeignKey('schedule.id', ondelete='CASCADE'), )
    schedule = relationship('Schedule', viewonly=True)

    gap_settings_id = Column(types.Integer, ForeignKey('manual_gap_settings.id', ondelete='CASCADE'), )
    gap_settings = relationship('ManualGapSettings', viewonly=True)


class ManualGap(TimestampedModelMixin, BaseModel):
    __tablename__ = 'manual_gap'

    id = Column(types.Integer, primary_key=True)

    gap_settings_id = Column(types.Integer, ForeignKey('manual_gap_settings.id', ondelete='CASCADE'), )
    gap_settings = relationship('ManualGapSettings', backref=backref('gaps', cascade='all,delete'))

    staff_id = Column(types.Integer, ForeignKey('intranet_staff.id', ondelete='CASCADE'), nullable=False)
    staff = relationship('Staff', foreign_keys=[staff_id], backref='manual_gaps')

    start = Column(types.DateTime(timezone=True))
    end = Column(types.DateTime(timezone=True))

    is_active = Column(types.Boolean, default=True)

    all_services = association_proxy(target_collection='gap_settings', attr='all_services')
    services = association_proxy(target_collection='gap_settings', attr='services')
    schedules = association_proxy(target_collection='gap_settings', attr='schedules')
