from typing import Tuple

from watcher.db import BaseModel, Interval, Slot, Shift
from watcher.api.schemas.base import BaseSchema
from watcher.api.schemas.slot import SlotPutSchema
from watcher.api.schemas.shift import ShiftPutSchema
from watcher.api.schemas.interval import IntervalPutSchema

INTERVAL_REVISION_RECALCULATED_FIELDS = ('duration', 'type_employment', 'unexpected_holidays', 'order')
SLOT_REVISION_RECALCULATED_FIELDS = ('composition_id', )
SHIFT_REVISION_RECALCULATED_FIELDS = ('staff_id', )


def recalculated_fields_changes(
    db_interval: Interval,
    interval: IntervalPutSchema,
) -> Tuple[bool, bool]:
    """
    Если изменились параметры интервала, нужно запустить таску на перерасчет границ
    Если изменились только составы слотов, то нужно запустить таску перераспределения людей

    :returns has_changes - есть изменения которые требуют перераспределения людей или пересчет границ
             need_revision_shift - нужно ли делать именно пересчет границ
    """

    has_changes = need_revision_shift = _has_recalculate_changes(
        obj=db_interval, schema=interval,
        fields_to_check=INTERVAL_REVISION_RECALCULATED_FIELDS,
    )

    exist_slots = {slot.id: slot for slot in db_interval.slots}
    if not has_changes and exist_slots:
        for slot in interval.slots:
            if recalculated_slot_fields_changes(exist_slots[slot.id], slot):
                has_changes = True
                need_revision_shift = False
                break

    return has_changes, need_revision_shift


def _has_recalculate_changes(obj: BaseModel, schema: BaseSchema, fields_to_check: Tuple[str, ...]) -> bool:
    update_data = schema.dict(exclude_unset=True)

    for attr in fields_to_check:
        current_value = getattr(obj, attr)

        if attr in update_data and current_value != update_data[attr]:
            return True

    return False


def recalculated_slot_fields_changes(db_slot: Slot, slot: SlotPutSchema) -> bool:
    return _has_recalculate_changes(db_slot, slot, SLOT_REVISION_RECALCULATED_FIELDS)


def recalculated_shift_fields_changes(db_shift: Shift, shift: ShiftPutSchema) -> bool:
    return _has_recalculate_changes(db_shift, shift, SHIFT_REVISION_RECALCULATED_FIELDS)
