import logging
from fastapi.responses import JSONResponse
from fastapi import Request
from fastapi.exceptions import RequestValidationError
from fastapi.encoders import jsonable_encoder
from starlette.status import HTTP_422_UNPROCESSABLE_ENTITY

from .exceptions import BaseWatcherException
from watcher.config import settings

logger = logging.getLogger(__name__)


async def base_exception_handler(request: Request, exc: BaseWatcherException) -> JSONResponse:
    """
    Обрабатывает все ошибки приложения и возвращает json с
    соответствующим кодом
    """
    return JSONResponse(
        status_code=exc.status_code,
        content=exc.to_json()
    )


async def unhandled_exception_handler(request: Request, exc: Exception) -> JSONResponse:
    logger.exception('Unhandled exception.')
    message = str(exc) if settings.ENV_TYPE != 'production' else 'Unhandled exception'
    return JSONResponse(
        status_code=500,
        content={'detail': message}
    )


async def request_validation_exception_handler_watcher(
    request: Request, exc: RequestValidationError
) -> JSONResponse:
    """
    Копия метода из fastapi, но с преобразованием
    msg: 'text' -> msg: {'ru': 'text', 'en': 'text'}
    """
    detail = jsonable_encoder(exc.errors())
    for error in detail:
        msg = error.get('msg')
        if msg and isinstance(msg, str):
            error['msg'] = {'ru': msg, 'en': msg}

    return JSONResponse(
        status_code=HTTP_422_UNPROCESSABLE_ENTITY,
        content={"detail": detail},
    )
