from watcher.config import settings


class BaseWatcherException(Exception):
    status_code = 500
    code = 'unknown_exception'
    default_message = {
        'ru': 'Необработанное исключение',
        'en': 'Unexpected exception',
    }

    def __init__(self, **kwargs):
        self.context = kwargs

    def __repr__(self):
        return f'{self.status_code} {self.code} -> {self.context}'

    def to_json(self):
        return {
            'error': self.code,
            'context': self.context,
            'detail': [
                {
                    'msg': self.context.get('message', self.default_message,)
                }
            ]
        }


class RecordNotFound(BaseWatcherException):
    status_code = 404
    code = 'not_found'
    default_message = {
        'ru': 'Объект не найден',
        'en': 'Object not found',
    }


class PermissionDenied(BaseWatcherException):
    status_code = 403
    code = 'permission_denied'
    default_message = {
        'ru': 'Доступ запрещен',
        'en': 'Permission denied',
    }


class NoValidAuthProvided(BaseWatcherException):
    status_code = 401
    code = 'unauthorized'
    default_message = {
        'ru': 'Авторизация неуспешна',
        'en': 'Unauthorized',
    }


class InvalidSecretToken(NoValidAuthProvided):
    default_message = {
        'ru': 'Передан невалидный X-SECRET-TOKEN',
        'en': 'Invalid X-SECRET-TOKEN',
    }


class LogbrokerException(BaseWatcherException):
    code = 'logbroker_exception'


class StaffNotFound(BaseWatcherException):
    status_code = 404
    code = 'staff_not_found'
    default_message = {
        'ru': 'Сотрудник не найден',
        'en': 'Staff member not found',
    }


class BadRequest(BaseWatcherException):
    status_code = 400
    code = 'bad_request'
    default_message = {
        'ru': 'Переданы невалидные данные',
        'en': 'Sent data is wrong',
    }


class ScheduleGroupHasReference(BadRequest):
    code = 'schedule_group_has_reference'
    default_message = {
        'ru': 'Есть связанные с группой расписания',
        'en': 'There are schedules associated with the group',
    }


class InvalidData(BaseWatcherException):
    status_code = 500
    code = 'invalid_data'

    default_message = {
        'ru': 'Неожиданная ошибка в данных',
        'en': 'Unexpected error in data',
    }


class RatingNotFound(InvalidData):
    code = 'rating_not_found'
    default_message = {
        'ru': 'У расписания нет рейтинга для сотрудника',
        'en': 'The schedule has no rating for the employee',
    }


class IsAutoGeneratedGroup(BadRequest):
    code = 'schedule_group_is_autogenerated'
    default_message = {
        'ru': 'В качестве группы расписаний нельзя указать автоcгенерированную группу',
        'en': 'An autogenerated group cannot be specified as a schedule group',
    }


class RecalculationInProcess(PermissionDenied):
    code = 'recalculation_in_process'
    default_message = {
        'ru': 'Редактирование расписания, которое сейчас обновляется невозможно',
        'en': 'Editing a schedule that is currently being updated is not possible',
    }


class ScheduleMaxDays(BadRequest):
    code = 'schedule_max_days'
    default_message = {
        'ru': f'Максимально допустимое значение - {settings.MAX_PROBLEM_OR_NOTIFICATION_DAYS} дней',
        'en': f'The maximum allowed value is {settings.MAX_PROBLEM_OR_NOTIFICATION_DAYS} days',
    }


class SlugMustBeUnique(BadRequest):
    code = 'schedule_slug_in_service_must_be_unique'
    default_message = {
        'ru': 'Slug расписания должен быть уникальным',
        'en': 'Schedule slug must be unique',
    }


class GroupSlugMustBeUnique(BadRequest):
    code = 'schedule_group_slug_must_be_unique'
    default_message = {
        'ru': 'Slug группы должен быть уникальным',
        'en': 'Group slug must be unique',
    }


class CompositionSlugMustBeUnique(BadRequest):
    code = 'composition_slug_must_be_unique'
    default_message = {
        'ru': 'Slug пула должен быть уникальным',
        'en': 'Pool slug must be unique',
    }


class SlugInvalidValue(BadRequest):
    code = 'slug_invalid_value'
    default_message = {
        'ru': 'Slug должен состоять из латинских букв, цифр, знаков подчеркивания или дефиса.',
        'en': 'Slug should contain only latin, numeric, underscore and hyphens characters.',
    }


class SlugInvalidLength(BadRequest):
    code = 'schedule_slug_invalid_length'
    default_message = {
        'ru': 'Минимальная длина слага - 3 символа, максимальная - 50. Поле обязательно.',
        'en': 'The minimum length of the slug is 3 character, the maximum is 50. The field is required.',
    }


class NameInvalidValue(BadRequest):
    code = 'name_invalid_value'
    default_message = {
        'ru': 'Минимальная длина названия - 1 символ, максимальная - 150. Поле обязательно.',
        'en': 'The minimum length of the name is 1 character, the maximum is 150. The field is required.',
    }


class ServiceMustBeActive(BadRequest):
    code = 'service_must_be_active'
    default_message = {
        'ru': 'Сервис должен находиться в активном статусе',
        'en': 'Service must be in the active status',
    }


class InvalidCompositionInput(BadRequest):
    code = 'invalid_composition_input'
    default_message = {
        'ru': 'Не указаны дежурные. Укажите роли, группы или логины для выбора дежурных',
        'en': 'The duty participants are not specified. Specify roles, groups, or logins to select duty participants',
    }


class NotResponsibleInAllServices(BadRequest):
    code = 'not_responsible_in_services'
    default_message = {
        'ru': 'Нужно являться управляющим во всех переданных сервисах',
        'en': 'You need to be a responsible in all specified services',
    }


class NoShiftsFound(RecordNotFound):
    code = 'no_shifts_found'


class ScheduleServiceNotInTheList(BadRequest):
    code = 'schedule_service_not_in_the_list'
    default_message = {
        'ru': 'Не все расписания соответствуют указанным сервисам',
        'en': 'Not all schedules correspond to the specified services',
    }


class ManualGapStartInPast(BadRequest):
    code = 'manual_gap_start_in_past'
    default_message = {
        'ru': 'Дата начала отсутствия должна быть не раньше сегодняшнего дня',
        'en': 'The start date of gap should not be earlier than today',
    }


class InvalidManualGapLength(BadRequest):
    code = 'invalid_manual_gap_length'
    default_message = {
        'ru': 'Недопустимая продолжительность отсутствия',
        'en': 'Invalid gap length',
    }


class WatcherAttributeError(BadRequest):
    code = 'attribute_error'

    def __init__(self, obj_name: str, attr_name: str, **kwargs):
        super().__init__(**kwargs)
        self.default_message = {
            'ru': f"У объекта '{obj_name}' нет атрибута '{attr_name}'",
            'en': f"Object '{obj_name}' has no attribute '{attr_name}'",
        }


class AbcMigrationException(BaseWatcherException):
    code = 'abc_migration_exception'


class BadRotatingIntervalSettings(BadRequest):
    code = 'bad_rotating_interval_settings'
    default_message = {
        'ru': 'В переданных сменах должен быть один пул дежурных, а также по одному запасному и основному дежурному.',
        'en': 'In the uploaded shifts there must be one same duty pool, as well as one backup and primary duty person in each.',
    }


class ShiftRatingDisabled(BadRequest):
    code = 'shift_rating_disabled'
    default_message = {
        'ru': 'Для этой смены рейтинг отключен.',
        'en': 'Rating is disabled for this shift.',
    }
