import operator
import logging
import collections

from typing import Optional

from fastapi import Query

from watcher.db import BaseModel
from watcher.config import settings
from watcher.logic.exceptions import WatcherAttributeError

logger = logging.getLogger(__name__)

OPERATOR_MAP = {
    'gt': operator.gt,
    'gte': operator.ge,
    'lt': operator.lt,
    'lte': operator.le,
}

REPLACER = 'B5B2CF77BBAEFCB1CC'


def get_filter_params(filter_params: Optional[str] = Query(None, alias='filter')) -> dict:
    if not filter_params:
        return {}

    result = {}
    in_queries = collections.defaultdict(set)
    or_counter = 0
    filter_params = filter_params.replace(r'\,', REPLACER)
    for item in filter_params.split(','):
        if '|' in item:
            result['|{}'.format(or_counter)] = get_filter_params(item.replace('|', ','))
            or_counter+=1
            continue
        field, value = item.split('=')
        value = value.replace(REPLACER, ',')
        if value in ('True', 'true'):
            value = True
        elif value in ('False', 'false'):
            value = False

        if field in result:
            # передан несколько раз одни и тот же ключ
            # считаем что нужно сделать in запрос
            in_queries[field].add(value)
        else:
            result[field] = value

    for field, values_set in in_queries.items():
        values_set.add(result.pop(field))
        result[f'{field}__in'] = values_set

    return result


def extract_joined_params(filter_params: dict) -> dict[str, dict[str, str]]:
    """
    Выделяет из переданного словаря - подсловари, с параметрами для связанных моделей
    filter_params - ожидается словарь вида
    {
        'id': 1,
        'service.owner.login': 'sa'
        'service.owner.is_robot': True
        'schedule.slug': 'abc'
    }
    на выход будет возвращен словарь вида
    {
        'service.owner': {
            'login': 'sa'
            'is_robot': True
        }
        'schedule': {
            'slug': 'abc'
        }
    }
    при этом в filter_params останутся только простые ключи
    {
        'id': 1,
    }
    """
    joined_model_params = collections.defaultdict(dict)
    for filter_key in list(filter_params):
        if '.' in filter_key:
            table, field = filter_key.rsplit('.', maxsplit=1)
            joined_model_params[table][field] = filter_params.pop(filter_key)
    return joined_model_params


def extract_filter_params_by_prefix(filter_params: dict, prefix: str) -> dict:
    """
    Выделяет из переданного словаря - отдельный словарь по переданному префиксу
    """
    extract_dict = {}
    for filter_key in list(filter_params):
        if filter_key.startswith(prefix):
            extract_dict[filter_key.removeprefix(prefix)] = filter_params.pop(filter_key)
    return extract_dict


def prepare_filter_params(model: BaseModel, filter_params: dict) -> list:
    """
    Подготавливает данные для их передачи в метод filter sqlalchemy

    Реализована поддержка следующих операторов:
    =,gt,lt,in,lte,gte,ilike

    """
    prepared_params = []
    for filter_key, value in filter_params.items():
        try:
            target_operator = operator.eq
            if '__' in filter_key:
                filter_key, operator_key = filter_key.split('__')

                if operator_key == 'in':
                    prepared_params.append(
                        getattr(getattr(model, filter_key), 'in_')(list(value))
                    )
                    continue
                if operator_key == 'ilike':
                    prepared_params.append(
                        getattr(
                            getattr(getattr(model, filter_key), 'ilike')(f'%{value}%'),
                            'collate'
                        )(settings.DEFAULT_COLLATION)
                    )
                    continue
                else:
                    target_operator = OPERATOR_MAP.get(operator_key)
                    if not target_operator:
                        logger.warning(f'Got unsupported operator: {filter_key}, value: {value}')
                        continue

            prepared_params.append(
                target_operator(getattr(model, filter_key), value)
            )

        except AttributeError:
            raise WatcherAttributeError(obj_name=model.__name__, attr_name=filter_key)
    return prepared_params
