import datetime

from sqlalchemy.orm import Session
from typing import Iterable, List

from watcher.crud.holiday import get_holidays, get_end_date_comparer
from watcher.db import Holiday


def get_extend_weekends(start: datetime.datetime, end: datetime.datetime) -> List[datetime.date]:
    cut_days = []
    while start.date() <= end.date():
        if start.date() == end.date() and end.time() == datetime.time(0, 0):
            # если конец смены в 00:00 последний день проверять не надо
            break
        if is_weekend(start):
            cut_days.append(start.date())
        start += datetime.timedelta(days=1)
    return cut_days


def is_weekday(date: datetime.date) -> bool:
    return date.weekday() < 5


def is_weekend(date: datetime.date) -> bool:
    return not is_weekday(date)


def is_holiday(session: Session, date: datetime.date) -> bool:
    return session.query(
        session.query(Holiday).filter(
            Holiday.date==date
        ).exists()
    ).scalar()


def get_unexpected_holidays(
    session: Session, start: datetime.datetime, end: datetime.datetime, holidays: Iterable[Holiday] = None
) -> List[Holiday]:
    """Получаем только 'внезапные' выходные/праздники за указанный период"""

    if holidays is None:
        holidays = get_holidays(db=session, start=start, end=end).all()

    result = [
        holiday for holiday in holidays
        if (
            start.date() <= holiday.date
            and get_end_date_comparer(check_date=holiday.date, end=end)
            and is_weekday(date=holiday.date)
        )
    ]

    return result


def get_unexpected_workday(
    session: Session, start: datetime.datetime, end: datetime.datetime, holidays: Iterable[Holiday] = None
) -> List[datetime.datetime]:
    """Получаем 'внезапные' рабочие дни за указанный период"""

    if holidays is None:
        holidays = get_holidays(db=session, start=start, end=end).all()

    holidays_dates = set(holiday.date for holiday in holidays)
    days = []
    while start.date() <= end.date():
        if start.weekday() >= 5 and start.date() not in holidays_dates:
            days.append(start)

        start += datetime.timedelta(days=1)

    return days
