from typing import Tuple, Dict

from watcher.api.schemas.interval import IntervalPutSchema
from watcher.api.schemas.slot import SlotListSchema
from watcher.db import Interval, Slot
from watcher import enums


def create_interval_slot(slot: SlotListSchema) -> Slot:
    slot_data = slot.dict()
    slot_data.pop('id')
    return Slot(**slot_data)


def create_interval(
    interval: IntervalPutSchema,
    **kwargs
) -> Interval:
    interval_data = interval.dict()
    interval_data.pop('slots')
    interval_data.pop('id')
    obj = Interval(**interval_data, **kwargs)

    # создать слот для пустого интервала
    if is_empty(obj):
        interval.slots.append(SlotListSchema())

    for slot in interval.slots:
        slot = create_interval_slot(slot)
        obj.slots.append(slot)

    return obj


def clone_interval_slot(slot: Slot, **kwargs) -> Slot:
    return Slot(
        points_per_hour=slot.points_per_hour,
        role_on_duty_id=slot.role_on_duty_id,
        composition_id=slot.composition_id,
        show_in_staff=slot.show_in_staff,
        **kwargs
    )


def clone_interval(db_interval: Interval, **kwargs) -> Tuple[Interval, Dict[int, Slot]]:
    new_interval = Interval(
        schedule_id=db_interval.schedule_id,
        duration=db_interval.duration,
        type_employment=db_interval.type_employment,
        unexpected_holidays=db_interval.unexpected_holidays,
        order=db_interval.order,
        **kwargs,
    )
    new_slots = {
        slot.id: clone_interval_slot(slot, interval=new_interval)
        for slot in db_interval.slots
    }

    return new_interval, new_slots


def is_empty(interval):
    return interval.type_employment == enums.IntervalTypeEmployment.empty


def check_rotation_validity(
    interval: Interval | IntervalPutSchema,
    expected_composition_id: int,
) -> bool:
    if interval.type_employment == enums.IntervalTypeEmployment.empty:
        return True
    slots = interval.slots
    if len(slots) != 2:
        return False
    if slots[0].is_primary == slots[1].is_primary:
        return False
    if not (slots[0].composition_id == slots[1].composition_id == expected_composition_id):
        return False
    return True
