import json
import gzip
import logging
import time

from typing import Generator
from concurrent.futures import TimeoutError
from io import BytesIO

import kikimr.public.sdk.python.persqueue.auth as auth
import kikimr.public.sdk.python.persqueue.grpc_pq_streaming_api as pqlib
from kikimr.public.sdk.python.persqueue.errors import SessionFailureResult

from watcher.config import settings
from .exceptions import LogbrokerException

logger = logging.getLogger(__name__)

LOGBROKER_MAX_TRIES = 3
DELAY = 3


class LogbrokerClient:
    def __init__(self, endpoint: str, topic: str, consumer: str):
        logger.info(f'Starting consumer: {endpoint}, {topic}, {consumer}')
        credentials_provider = auth.OAuthTokenCredentialsProvider(
            settings.LOGBROKER_TOKEN,
        )
        self.api = pqlib.PQStreamingAPI(
            endpoint,
            settings.LOGBROKER_PORT,
        )
        api_start_future = self.api.start()
        api_start_future.result(timeout=10)

        configurator = pqlib.ConsumerConfigurator(
            topic,
            consumer,
        )

        self.consumer = self.api.create_consumer(
            configurator,
            credentials_provider=credentials_provider,
        )

        start_future = self.consumer.start()
        start_result = start_future.result(timeout=10)

        if not isinstance(start_result, SessionFailureResult):
            if start_result.HasField('init'):
                logger.info(f'Consumer start result was: {start_result}')
            else:
                raise LogbrokerException(error=f'Bad consumer start result from server: {start_result}')
        else:
            raise LogbrokerException(error=f'Error occurred on start of consumer: {start_result}')
        logger.info(f'Consumer started: {endpoint}, {topic}, {consumer}')

    def get_data(self) -> Generator['kikimr.yndx.api.protos.persqueue_pb2.Message', None, None]:
        # Cookies are used for commits. Commits confirm we read and processed the message.
        # You may either commit every single cookie when all messages from batch are processed or collect several cookies
        # and commit them in one pack. In any scenario cookie must be kept before you commit it.
        # WARNING: Loosing cookies is a definitely incorrect behavior and will bring just pain
        # Last cookie we got
        last_received_cookie = 1
        # Last cookie committed and acked by server
        last_committed_cookie = 0
        total = 0

        while last_received_cookie != last_committed_cookie:
            try:
                result = self.consumer.next_event().result(timeout=5)
            except TimeoutError:
                # no message found
                break

            if result.type == pqlib.ConsumerMessageType.MSG_DATA:
                for batch in result.message.data.message_batch:
                    for message in batch.message:
                        yield message
                        total += 1
                # Now commit the cookie to confirm we got and processed this message
                self.consumer.commit(result.message.data.cookie)
                last_received_cookie = result.message.data.cookie
            else:
                # No other message types are expected in this sample
                assert result.type == pqlib.ConsumerMessageType.MSG_COMMIT
                last_committed_cookie = result.message.commit.cookie[-1]

        logger.info(f'Got {total} messages in total')
        self.shutdown()

    def shutdown(self):
        logger.info('Stopping client')
        self.consumer.stop()
        self.api.stop()
        logger.info('Stopped client')


def process_message(message) -> Generator[tuple, None, None]:
    data_stream = BytesIO(message.data)
    gzip_file_handle = gzip.GzipFile(fileobj=data_stream)
    data = json.loads(gzip_file_handle.read())
    for item in data:
        table = item['table']
        object_data = {}
        if table in settings.DATA_TRANSFER_SYSTEM_TABLES:
            # события в этих таблицах нам не интересны
            continue
        kind = item['kind']
        columns = item.get('columnnames')
        values = item.get('columnvalues')
        if columns and values:
            object_data = dict(zip(columns, values))
        oldkeys = item.get('oldkeys', {})
        if oldkeys:
            oldkeys = dict(zip(oldkeys['keynames'], oldkeys['keyvalues']))

        yield table, kind, object_data, oldkeys


def get_new_data_from_topic(topic: str, consumer: str) -> Generator[tuple, None, None]:
    for endpoint in settings.LOGBROKER_ENDPOINTS:
        for n_try in range(1, LOGBROKER_MAX_TRIES):
            try:
                client = LogbrokerClient(
                    endpoint=endpoint,
                    topic=topic,
                    consumer=consumer,
                )
                break
            except TimeoutError as exc:
                if n_try < LOGBROKER_MAX_TRIES:
                    time.sleep(DELAY)
                    continue
                raise exc

        data = client.get_data()

        for message in data:
            processes = process_message(message)
            for item in processes:
                yield item
