from typing import Optional
from dateutil import rrule
import datetime

from watcher.db import (
    ManualGapSettings,
)
from watcher.logic.timezone import now
from watcher.enums import ManualGapRecurrence
from watcher.config import settings


def generate_gaps(
    gap_settings: ManualGapSettings,
    last_start_date: Optional[datetime.datetime] = None,
    from_datetime: Optional[datetime.datetime] = None,
    to_datetime: Optional[datetime.datetime] = None
) -> list[dict]:
    """
    Создает гэпы по переданным настройкам
    :param gap_settings:
    :param last_start_date: начало последнего созданного гэпа по этим настройкам,
        если не указано, то берется gap_settings.start. Используется для определения start первого нового гэпа
    :param from_datetime: начало расчета
    :param to_datetime: конец расчета
    :return:
    """
    current_now = now()
    from_datetime = from_datetime or current_now
    to_datetime = to_datetime or current_now + datetime.timedelta(days=60)
    to_create = list()

    if not gap_settings.is_active:
        return []

    dates = list()
    if gap_settings.recurrence is ManualGapRecurrence.once:
        if from_datetime < gap_settings.end:
            dates.append(gap_settings.start)
    else:
        kwargs, timedelta = settings.GAP_RECURRENCE_TO_DT[gap_settings.recurrence]
        dtstart = last_start_date + timedelta if last_start_date else gap_settings.start
        for dt in rrule.rrule(
            dtstart=dtstart, until=to_datetime, **kwargs
        ):
            if dt < from_datetime:
                continue
            dates.append(dt)

    gap_length = gap_settings.end - gap_settings.start
    for date in dates:
        to_create.append({
            'staff_id': gap_settings.staff_id,
            'gap_settings_id': gap_settings.id,
            'start': date,
            'end': date+gap_length,
        })
    return to_create


def is_applicable(gap, schedule_id: int, service_id: int):
    try:
        schedules = getattr(gap, '_set_schedules')
        services = getattr(gap, '_set_services')
    except AttributeError:
        schedules = {obj.id for obj in gap.schedules}
        services = {obj.id for obj in gap.services} if not gap.schedules else set()
        setattr(gap, '_set_schedules', schedules)
        setattr(gap, '_set_services', services)
    return (
        gap.all_services
        or (schedule_id in schedules)
        or (not schedules and service_id in services)
    )
