import logging

from sqlalchemy.orm import Session

from watcher import enums
from watcher.crud.member import get_member_by_shift
from watcher.crud.shift import get_shifts_with_same_role
from watcher.crud.notification import create_shift_starting_now_notification
from watcher.db import Shift
from watcher.logic.clients.abc import abc_client

logger = logging.getLogger(__name__)


def check_and_request_role(session: Session, shift: Shift):
    if not shift.slot or not shift.slot.role_on_duty_id:
        shift.status = enums.ShiftStatus.active
        create_shift_starting_now_notification(db=session, shift=shift)
        logger.info(f'Started shift without role: {shift.id}')

    else:
        member = get_member_by_shift(db=session, shift=shift)
        if not member:
            abc_client.request_role(
                service=shift.schedule.service_id,
                login=shift.staff.login,
                role=shift.slot.role_on_duty_id,
            )
            shift.is_role_requested = True
        else:
            logger.info(
                f'Already has duty role, shift: {shift.id}, '
                f'role: {shift.slot.role_on_duty_id}, '
                f'staff: {shift.staff.login}'
            )
        create_shift_starting_now_notification(db=session, shift=shift)
        shift.status = enums.ShiftStatus.active


def check_and_deprive_member(session: Session, shift: Shift) -> None:
    if not shift.staff_id or not shift.is_role_requested:
        return

    member = get_member_by_shift(db=session, shift=shift)
    if member:
        # проверим - не идет ли сейчас смена с такой же ролью
        shifts_with_same_role = get_shifts_with_same_role(
            db=session, shift=shift,
        )
        if shifts_with_same_role:
            logger.info(f'Find shifts with same role, not depriving {member.id}')
            # обновим смены, чтобы когда они закончатся - роль все же отозвалась
            session.query(Shift).filter(
                Shift.id.in_(obj.id for obj in shifts_with_same_role)
            ).update(
                {Shift.is_role_requested: True},
                synchronize_session=False,
            )
        else:
            abc_client.deprive_role(
                membership_id=member.id,
            )
    else:
        logger.warning(
            f'No such member found: shift: {shift.id}, '
            f'role: {shift.slot.role_on_duty_id}, '
            f'staff: {shift.staff.login}'
        )
