import hashlib

from sqlalchemy.orm import Session

from watcher.logic.timezone import now
from watcher.config import settings
from watcher.crud.bot_user import query_bot_user_by_staff_ids
from watcher.db import Notification
from watcher import enums


def get_notification_channels(db: Session, staff_ids: list[int]) -> dict:
    bot_users = query_bot_user_by_staff_ids(db=db, staff_ids=staff_ids)
    staff_with_telegram = {bot_user.staff_id for bot_user in bot_users}
    return {
        staff_id:
            enums.JnsChannel.telegram if staff_id in staff_with_telegram
            else enums.JnsChannel.email
        for staff_id in staff_ids
    }


def get_common_params(notification: Notification) -> dict:
    schedule = notification.shift.schedule
    return {
        'duty_name': {
            'string_value': schedule.name
        },
        'duty_link': {
            'string_value': f'https://{settings.ABC_HOST}/services/{schedule.service.slug}/duty2/{schedule.id}/'
        },
        'service_name': {
            'string_value': schedule.service.name
        },
        'service_link': {
            'string_value': f'https://{settings.ABC_HOST}/services/{schedule.service.slug}/'
        },
    }


def get_start_shift_soon_params(notification: Notification) -> dict:
    params = get_common_params(notification=notification)
    params['date'] = {'string_value': notification.shift.start.date().isoformat()}
    params['time'] = {'string_value': notification.shift.start.strftime('%H:%M')}
    params['shift_type'] = {
        'string_value': 'primary-дежурство' if notification.shift.is_primary else 'backup-дежурство',
    }
    return params


def get_start_shift_params(notification: Notification) -> dict:
    params = get_common_params(notification=notification)
    params['shift_type'] = {
        'string_value': 'primary-дежурство' if notification.shift.is_primary else 'backup-дежурство',
    }
    return params


def get_problem_nobody_on_duty_params(notification: Notification) -> dict:
    params = get_common_params(notification=notification)
    params['shift_start'] = {
        'string_value': notification.shift.start.strftime('%Y-%m-%d %H:%M:%S')
    }
    params['shift_end'] = {
        'string_value': notification.shift.end.strftime('%Y-%m-%d %H:%M:%S')
    }
    return params


def get_problem_staff_has_gap_params(notification: Notification) -> dict:
    params = get_common_params(notification=notification)
    staff = notification.shift.staff
    params['staff_name'] = {
        'string_value': ' '.join([staff.first_name, staff.last_name])
    }
    params['staff_link'] = {
        'string_value': f'https://staff.yandex-team.ru/{staff.login}/'
    }
    params['shift_start'] = {
        'string_value': notification.shift.start.strftime('%Y-%m-%d %H:%M:%S')
    }
    params['shift_end'] = {
        'string_value': notification.shift.end.strftime('%Y-%m-%d %H:%M:%S')
    }
    return params


def get_hash(*args) -> str:
    hash_params = ''.join(map(str, args))
    return hashlib.sha256(hash_params.encode()).hexdigest()


def get_start_shift_request_id(notification: Notification) -> str:
    return get_hash('shift', notification.shift_id, notification.staff_id)


def get_start_shift_soon_request_id(notification: Notification) -> str:
    return get_hash(
        'shift',
        notification.shift_id,
        notification.staff_id,
        (notification.shift.start - now()).days
    )


def get_problem_nobody_on_duty_request_id(notification: Notification) -> str:
    return get_hash(
        'problem', enums.ProblemReason.nobody_on_duty,
        notification.shift_id, notification.staff_id,
    )


def get_problem_staff_has_gap_request_id(notification: Notification) -> str:
    return get_hash(
        'problem', enums.ProblemReason.staff_has_gap,
        notification.shift_id, notification.staff_id,
    )
