from typing import Optional, Tuple
from base64 import b64decode, b64encode
from urllib.parse import urlencode, parse_qsl, unquote

from fastapi import Query, Request
from sqlalchemy.orm.query import Query as SqlQuery

from watcher.db.base import BaseModel
from watcher.config import settings
from watcher.logic.filter import prepare_filter_params
from watcher.logic.order import prepare_order_by_params


def get_cursor_param(cursor: Optional[str] = Query(None)) -> Optional[str]:
    return cursor


def get_page_size_param(
    page_size: Optional[int] = Query(
        default=settings.PAGE_SIZE,
        ge=1,
        le=settings.MAX_PAGE_SIZE,
    )
) -> int:
    return page_size


class CursorPagination:
    CURSOR_PARAM = 'cursor'
    REVERSE_PARAM = 'reverse'

    def __init__(
        self, cursor: str, page_size: int,
        query: SqlQuery, model: BaseModel,
        request: Request, ordering: tuple,
    ):
        self.cursor = cursor
        self.page_size = page_size
        self.query = query
        self.model = model
        self.request = request
        self.ordering = ordering

    def _get_cursor_directions(self, fields: dict) -> dict:
        result = {}
        for field, value in fields.items():
            order = 'asc'
            if field.startswith('-'):
                order = 'desc'
                field = field[1:]
            result[field] = {'direction': order, 'value': value}

        return result

    def _get_filter_params(self, cursor_directions: dict, is_reverse: bool) -> dict:
        """
        Подготавливает данные для построения запроса к базе
        на основе данных курсора
        """
        filter_params = {}
        directions = {'asc': 'gt', 'desc': 'lt'}
        if is_reverse:
            directions = {'desc': 'gt', 'asc': 'lt'}

        for field, data in cursor_directions.items():
            direction = data['direction']
            field_value = data['value']
            oper = directions[direction]
            filter_params[f'{field}__{oper}'] = field_value

        return filter_params

    @staticmethod
    def _encode_cursor(cursor_data: dict) -> str:
        querystring = urlencode(cursor_data, doseq=True)
        return b64encode(querystring.encode('ascii')).decode('ascii')

    @staticmethod
    def _decode_cursor(cursor: str) -> dict:
        cursor = unquote(cursor)
        querystring = b64decode(cursor.encode('ascii')).decode('ascii')
        decoded = parse_qsl(querystring, keep_blank_values=True)
        return {
            key: value
            for key, value in decoded
        }

    def _get_cursor_values(self, item: BaseModel) -> dict:
        """
        Получаем из объекта значения для передачи в курсоре
        в next/prev ссылках
        """
        result = {}
        for order_field in self.ordering:
            field_key = order_field
            if field_key.startswith('-'):
                field_key = field_key[1:]
            value = getattr(item, field_key)
            result[order_field] = value

        return result

    def _get_cursor_link(self, cursor_data: dict) -> str:
        encoded_cursor = self._encode_cursor(cursor_data)
        return str(self.request.url.include_query_params(
            **{self.CURSOR_PARAM: encoded_cursor}
        )).replace('http://', 'https://')

    def get_links(self, items: list, is_last: bool, is_reverse: bool) -> Tuple[str, str]:
        next_link = None
        if not is_last or (is_last and is_reverse):
            next_cursor_data = self._get_cursor_values(items[-1])
            next_link = self._get_cursor_link(next_cursor_data)

        prev_link = None
        if self.cursor and not (is_last and is_reverse):
            prev_cursor_data = self._get_cursor_values(items[0])
            prev_cursor_data[self.REVERSE_PARAM] = 1
            prev_link = self._get_cursor_link(prev_cursor_data)

        return next_link, prev_link

    def _limit_query_results(self) -> Tuple[SqlQuery, bool]:
        query = self.query
        is_reverse = False

        if self.cursor:
            decoded_cursor = self._decode_cursor(self.cursor)
            is_reverse = bool(decoded_cursor.pop(self.REVERSE_PARAM, 0))
            cursor_directions = self._get_cursor_directions(decoded_cursor)

            cursor_params = self._get_filter_params(
                cursor_directions,
                is_reverse=is_reverse,
            )
            filter_params = prepare_filter_params(
                filter_params=cursor_params,
                model=self.model,
            )
            query = query.filter(*filter_params)

        return query, is_reverse

    def _order_query(self, query: SqlQuery, is_reverse: bool) -> SqlQuery:
        order_params = prepare_order_by_params(
            model=self.model, ordering=self.ordering,
            is_reverse=is_reverse,
        )
        return query.order_by(*order_params)

    def paginate(self) -> Tuple[str, str, list]:
        query, is_reverse = self._limit_query_results()
        query = self._order_query(query=query, is_reverse=is_reverse)

        # запрашиваем на один больше, чтобы понять
        # есть ли следующая страница
        items = query.limit(self.page_size + 1).all()

        is_last = len(items) <= self.page_size
        if not is_last:
            del items[-1]

        if is_reverse:
            # тут логика такая - когда пользователь приходит по prev ссылке
            # то сортировка происходит в обратно направлении (нужно для корректного
            # отбора элементов) и в результате этого items оказывается развернут
            # в обратную сторону, тут приводим его в нужный порядок
            items.reverse()

        next_link, prev_link = self.get_links(
            items=items,
            is_reverse=is_reverse,
            is_last=is_last,
        )

        return next_link, prev_link, items
