from collections import defaultdict
from typing import List, Iterable, Optional

from sqlalchemy.orm import Session

from watcher.crud.permissions import has_perm
from watcher.config import settings
from watcher.db import (
    Staff, Schedule,
    ScheduleResponsible,
    ScheduleGroupResponsible,
    Member, Role,
    Problem,
)
from watcher.logic.service import (
    is_responsible_in_service,
    is_user_in_service,
    get_ancestors_map,
)
from watcher.enums import MemberState
from watcher.crud.service import get_services_by_ids


def has_full_access(db: Session, user_id: int) -> bool:
    return has_perm(db=db, user_id=user_id, permission_ids=[settings.FULL_ACCESS_ID])


def has_limited_role(db: Session, user_id: int) -> bool:
    return has_perm(db=db, user_id=user_id, permission_ids=[settings.SERVICES_VIEWER_ID])


def has_strictly_or_limited_or_full_role(db: Session, user_id: int) -> bool:
    return has_perm(
        db=db, user_id=user_id,
        permission_ids=[
            settings.OWN_ONLY_VIEWER_ID,
            settings.SERVICES_VIEWER_ID,
            settings.FULL_ACCESS_ID,
        ]
    )


def has_strictly_limited_role(db: Session, user_id: int) -> bool:
    return has_perm(db=db, user_id=user_id, permission_ids=[settings.OWN_ONLY_VIEWER_ID])


def is_group_responsible(db: Session, group_id: int, staff: Staff) -> bool:
    if is_superuser(staff=staff):
        return True
    return db.query(db.query(ScheduleGroupResponsible).filter(
        ScheduleGroupResponsible.schedule_group_id == group_id,
        ScheduleGroupResponsible.responsible_id == staff.id
    ).exists()).scalar()


def is_schedule_responsible(db: Session, schedule: Schedule, staff_id: int) -> bool:
    """
    Проверяет, является ли переданный staff_id ответственным за schedule
    """
    return db.query(db.query(ScheduleResponsible).filter(
        ScheduleResponsible.schedule_id == schedule.id,
        ScheduleResponsible.responsible_id == staff_id
    ).exists()).scalar()


def is_user_in_service_or_responsible(db: Session, staff: Staff, service_id: int) -> bool:
    """
    Проверка что человек принадлежит сервису, или является ответственным в одном из родительских сервисов
    """
    if not is_superuser(staff=staff):
        if not is_user_in_service(db=db, staff_id=staff.id, service_id=service_id):
            if not is_responsible_in_service(db=db, staff_id=staff.id, service_id=service_id):
                return False
    return True


def is_user_responsible_for_service_or_schedule(db: Session, schedule: Schedule, staff: Staff) -> bool:
    """
    Проверка что человек является ответственным в одном из родительских сервисов (включая переданный)
    или состоит в переданном сервисе и является ответственным за переданное расписание
    """
    if (
        is_superuser(staff=staff)
        or is_responsible_in_service(db=db, staff_id=staff.id, service_id=schedule.service_id)
        or (
            is_user_in_service(db=db, staff_id=staff.id, service_id=schedule.service_id)
            and is_schedule_responsible(db=db, schedule=schedule, staff_id=staff.id)
        )
    ):
        return True
    return False


def is_superuser(staff: Staff):
    return staff.login in settings.SUPERUSERS


def schedules_responsible_in(db: Session, staff: Staff, schedules: List[Schedule]) -> set[int]:
    """
    Возвращает список id расписаний среди переданных, где пользователь
    является ответственным
    """
    schedules_ids = {schedule.id for schedule in schedules}
    if is_superuser(staff=staff):
        return schedules_ids

    # ответственным за какие расписания является staff_id
    schedules_responsibles_id = {
        schedule_id for schedule_id, in
        db.query(ScheduleResponsible.schedule_id).filter(
            ScheduleResponsible.responsible_id == staff.id,
            ScheduleResponsible.schedule_id.in_(schedules_ids)
        )
    }
    if schedules_responsibles_id == schedules_ids:
        return schedules_ids

    # узнаем ancestors для каждого раписания из schedules
    schedule_ancestors = {
        schedule.id: set(
            [schedule.service.id] + [
                ancestor['id'] for ancestor in schedule.service.ancestors
                if 'id' in ancestor
            ]
        ) for schedule in schedules if
        schedule.id not in schedules_responsibles_id
    }

    service_ids = {
        ancestor_id for ancestors in schedule_ancestors.values()
        for ancestor_id in ancestors
    }

    # все Member, где staff_id имеет роль ответственного для всех ancestors
    ancestors_responsible_members = db.query(Member.service_id).join(Role, Role.id == Member.role_id).filter(
        Member.staff_id == staff.id,
        Member.state == MemberState.active,
        Member.service_id.in_(service_ids),
        Role.code.in_(Role.WATCHER_RESPONSIBLES),
    ).all()
    responsible_in_ancestors = {service_id for service_id, in ancestors_responsible_members}

    # для каждого расписания узнаем, есть ли у staff_id роль в его ancestors
    for schedule_id, ancestors in schedule_ancestors.items():
        if responsible_in_ancestors.intersection(ancestors):
            schedules_responsibles_id.add(schedule_id)

    return schedules_responsibles_id


def services_responsible_in(db: Session, services_ids: Iterable[int], staff: Staff) -> set[int]:
    """
    Возвращает список сервисов среди переданных, где указанный пользователь
    является ответственным, учитывая так же роли в родительских сервисах
    """

    if is_superuser(staff=staff):
        return set(services_ids)

    services_map = get_ancestors_map(
        services=get_services_by_ids(
            db=db,
            services_ids=services_ids,
        )
    )

    ancestors_ids = {
        service_id for services in services_map.values()
        for service_id in services
    }

    services_with_ancestors_responsible_in = {
        service_id for service_id, in
        (db.query(Member.service_id)
         .join(Role, Role.id == Member.role_id)
         .filter(
            Member.staff_id == staff.id,
            Member.state == MemberState.active,
            Member.service_id.in_(ancestors_ids),
            Role.code.in_(Role.WATCHER_RESPONSIBLES),
        )
        )
    }

    services_responsible_in = set()
    for service_id, ancestors in services_map.items():
        if ancestors.intersection(services_with_ancestors_responsible_in):
            services_responsible_in.add(service_id)

    return services_responsible_in


def get_problems_responsibles(db: Session, problems: list[Problem]) -> dict[Problem, Optional[set[int]]]:
    """
    Возвращает мапу проблем к множеству staff.id ответственных за график,
    если их нет то к управляющим дежурствами,
    если и их нет - то к управляющим сервисом
    """

    schedule_ids = {problem.shift.schedule_id for problem in problems}
    schedule_responsibles_query = db.query(ScheduleResponsible).filter(
        ScheduleResponsible.schedule_id.in_(schedule_ids)
    )
    schedule_responsibles = defaultdict(set)

    for responsible in schedule_responsibles_query:
        schedule_responsibles[responsible.schedule_id].add(responsible.responsible_id)

    service_ids = {problem.shift.schedule.service_id for problem in problems}
    service_responsibles_query = db.query(Member, Role.code).join(
        Role, Role.id == Member.role_id
    ).filter(
        Member.service_id.in_(service_ids),
        Member.state == MemberState.active,
        Role.code.in_(Role.WATCHER_RESPONSIBLES),
    )
    service_responsibles = defaultdict(lambda : defaultdict(set))

    for responsible_member, role_code in service_responsibles_query:
        service_responsibles[responsible_member.service_id][role_code].add(
            responsible_member.staff_id
        )

    problem_responsibles = dict()
    for problem in problems:
        schedule_id = problem.shift.schedule_id
        service_id = problem.shift.schedule.service_id

        if schedule_id in schedule_responsibles:
            problem_responsibles[problem] = schedule_responsibles[schedule_id]
        elif service_id in service_responsibles:
            for role_code in Role.WATCHER_RESPONSIBLES:
                staff_ids = service_responsibles[service_id][role_code]
                if staff_ids:
                    problem_responsibles[problem] = staff_ids
                    break

    return problem_responsibles
