import datetime

from sqlalchemy.orm import Session, joinedload

from watcher import enums
from watcher.crud.revision import query_active_revisions
from watcher.crud.base import commit_object
from watcher.db import Revision
from watcher.logic.timezone import localize


def create_schedule_revision(
    db: Session,
    schedule_id: int,
    apply_datetime: datetime.datetime,
) -> Revision:
    """
    Создаем новую ревизию, так же ищем среди ревизий упорядоченных
    по apply_datetime ей подходящее место.
    """
    apply_datetime = localize(apply_datetime)

    schedule_revisions = query_active_revisions(
        db=db, schedule_id=schedule_id,
    ).options(
        joinedload(Revision.next, Revision.prev)
    ).order_by(Revision.apply_datetime.asc()).all()

    revision = Revision(
        schedule_id=schedule_id,
        apply_datetime=apply_datetime,
    )
    revision = commit_object(db=db, obj=revision)

    for obj in schedule_revisions:
        if not obj.next:
            # дошли до конца последовательности ревизий
            if obj.apply_datetime > apply_datetime:
                if obj.prev:
                    obj.prev.next = revision

                revision.next = obj
            else:
                obj.next = revision
            break

        elif obj.apply_datetime <= apply_datetime <= obj.next.apply_datetime:
            revision.next_id = obj.next_id
            obj.next_id = revision.id
            break

    # фиксируем изменения в измененных ревизиях
    if schedule_revisions:
        db.commit()

    return revision


def remove_revision(revision: Revision) -> None:
    """
    Убираем ревизию из последовательности и деактивируем ее
    """
    revision.state = enums.RevisionState.disabled
    if revision.prev:
        revision.prev.next = revision.next
    revision.next = None
