from typing import Iterable
from sqlalchemy.orm import Session

from watcher.logic.exceptions import BadRequest
from watcher.db import Role, Member, Staff, Service
from watcher.crud.base import list_objects_by_model
from watcher.crud.member import query_members_in_service_by_staff
from watcher.enums import MemberState


def check_roles_service(db: Session, roles_ids: Iterable, expected: int) -> None:
    """
    Проверяет что все переданные роли либо глобальные,
    либо относятся к переданному сервису
    """
    roles_from_another_service = (
        list_objects_by_model(db=db, model=Role)
        .filter(
            Role.id.in_(roles_ids),
            Role.service_id.isnot(None),
            Role.service_id!=expected,
        )
    ).all()
    if roles_from_another_service:
        raise BadRequest(
            message={
                'ru': 'Не все роли принадлежат данному сервису',
                'en': 'Not all roles belong to this service',
            },
            roles_from_another_service=[
                role.name for role in roles_from_another_service
            ],
        )


def check_staff_service(db: Session, staff_ids: Iterable, expected: int) -> None:
    """
    Проверяем, что переданные люди состоят в данном сервисе
    """

    staff_ids = set(staff_ids)
    members_in_service = query_members_in_service_by_staff(db, staff_ids=staff_ids, service_id=expected)
    staff_in_service_ids = {member.staff_id for member in members_in_service}

    if staff_ids != staff_in_service_ids:
        missing_staff_logins = (
            db.query(Staff.login)
            .filter(
                Staff.id.in_(staff_ids.difference(staff_in_service_ids))
            )
        )
        raise BadRequest(
            message={
                'ru': 'Не все переданные сотрудники состоят в сервисе',
                'en': 'Not all employees are members of service',
            },
            missing_logins=[obj_login for (obj_login,) in missing_staff_logins],
        )


def check_staff_responsible_in_services(db: Session, staff_id: int, service_ids: list[int]) -> bool:
    """
    Проверяем, имеет ли человек роль управляющего или управляющего дежурствами в любом из переданных сервисов
    """
    service_ids = set(service_ids)

    return db.query(
        db.query(Member.id)
        .join(Role, Role.id == Member.role_id)
        .filter(
            Member.staff_id == staff_id,
            Member.state == MemberState.active,
            Member.service_id.in_(service_ids),
            Role.code.in_(Role.WATCHER_RESPONSIBLES),
        )
        .exists()
    ).scalar()


def is_responsible_in_service(db: Session, staff_id: int, service_id: int = None) -> bool:
    """
    Проверяет является ли переданный staff_id управляющим сервиса/дежурствами
    его или его потомков
    """
    ancestors = db.query(Service).get(service_id).ancestors
    service_ids = [service_id]
    service_ids.extend([ancestor['id'] for ancestor in ancestors if 'id' in ancestor])
    return check_staff_responsible_in_services(db=db, staff_id=staff_id, service_ids=service_ids)


def is_user_in_service(db: Session, staff_id: int, service_id: int):
    return db.query(
        query_members_in_service_by_staff(db, staff_ids=[staff_id], service_id=service_id).exists()
    ).scalar()


def get_ancestors_map(services: list[Service]) -> dict[int, set]:
    return {
        service.id: set(
            [service.id] + [
                ancestor['id'] for ancestor in service.ancestors
                if 'id' in ancestor
            ]
        ) for service in services
    }
