import copy
import datetime
import logging
from typing import List, Dict, Tuple, Iterable, Optional

from sqlalchemy.orm import Session

from watcher import enums
from watcher.api.schemas.shift import ShiftPutSchema
from watcher.crud.shift import query_shifts_without_next, query_shifts_with_intervals_by_schedule
from watcher.crud.staff import get_staff_by_id
from watcher.db import Shift
from watcher.logic.member import check_and_deprive_member
from watcher.logic.rating import update_staff_ratings_for_further_shifts, get_predicted_ratings_after_shift
from watcher.logic.timezone import make_localized_datetime, localize, now
from watcher.logic.rating import update_ratings_for_past_shift

logger = logging.getLogger(__name__)


class TimeRange:
    def __init__(self, start, end):
        self.start = start
        self.end = end


def bind_sequence_shifts(sequence_shifts: List[Shift]) -> None:
    for i, shift in enumerate(sequence_shifts):
        if i + 1 < len(sequence_shifts):
            shift.next = sequence_shifts[i + 1]

            if shift.end != sequence_shifts[i + 1].start and shift.end != sequence_shifts[i + 1].end:
                # тк шифты могут быть параллельными
                logger.error('Последовательность шифтов обрывается')


def repair_sequence_shifts(db: Session, schedule_id: int) -> None:
    first_broken_shift = query_shifts_without_next(db, schedule_id).order_by(Shift.end).first()
    generate_new_sequence_by_shift(db, first_broken_shift)


def add_one_subshift(time_range, start, end, shift, empty=False) -> Shift:
    """
    Формирование подсмены для внезапных дней (рабочих/выходных)
    с опредление старта и конца

    :param time_range: диапазон, на который формируем замену.
        - В зависимости от переданного параметра empty
            это может быть внезапный выходной или наоборот внезапный рабочий.
            Например, если нужно сформировать замену на внезапный праздник,
            предается день праздника и параметр empty == True.
    :param start: предположительный старт замены,
        - Поскольку в смене может быть несколько замен, или сама замена может быть в середине смены,
            то старт, с которого стоит проверять может отличаться от старта основной смены.
            Т.е. фактически это точка, ранее которой старт у замены быть не может.
        - В формируемой в данном методе замене стартом будет максимальный datetime
            из начала дня (праздника/внезапного рабочего) или из переданного значения старта
        - Допустим, мы формируем первый внезапный выходной, который выпадет на середину смены,
            тогда сюда будет передан старт смены. Т.к. замена должна быть на более позднюю дату,
            то в шифт запишем старт выходного дня (он будет максимальным среди двух дат).
            Если смена начинается в момент внезапного выходного, в старт так же будет передан старт шифта
            и в смене запишем максимальную дату - это старт смены, тк замена не может начинаться раньше самой смены.
    :param end: предполагаемый конец смены
        - Аналогично старту подсмена не может, например, закончится после конца смены.
            Поэтому в смену запишется минимальная дата из предполагаемого окончания
            и окончания внезапного рабочего/выходного дня
    :param shift: шифт, к которому будет относится замена
    :param empty: признак пустоты замены
    :return: объект подсмены, но ещё не сохраненный в базе
    """

    return Shift(
        start=max(
            time_range.start,
            localize(start)
        ),
        end=min(
            time_range.end,
            localize(end)
        ),
        schedule_id=shift.schedule_id,
        replacement_for=shift,
        slot_id=shift.slot_id,
        empty=empty,
        is_primary=shift.is_primary,
    )


def merge_days(days: List[datetime.date]) -> List[TimeRange]:
    """
    Объединяем соседствующие дни в диапазоны.
    :param days: список дней,
    :return: список диапазонов
    """
    merged_days = []
    for day in sorted(days):
        localized_day = make_localized_datetime(day)
        localized_next_day = make_localized_datetime(day + datetime.timedelta(days=1))
        if merged_days and merged_days[-1].end == localized_day:
            merged_days[-1].end = localized_next_day
        else:
            merged_days.append(TimeRange(start=localized_day, end=localized_next_day))
    return merged_days


def add_subshifts(cut_days: List[datetime.date], shift: Shift) -> List[Shift]:
    """
    По списку переданных дней сформируем смены-замены.
    :param cut_days: список вырезаемых дней,
    :param shift:
    :return: список замен, которые будут сформированы, но не закомичены
    """
    result = []
    start = localize(shift.start)
    for time_range in merge_days(cut_days):
        start_range = time_range.start
        if start < start_range:
            # значит до выходного есть кусок, который нужно отдежурить (в случае, если переданы holidays)
            # добавляем вначале замену, которая будет с человеком
            # тип занятости данной подсмены совпадает с основным шифтом
            result.append(
                Shift(
                    start=start,
                    end=start_range,
                    schedule_id=shift.schedule_id,
                    replacement_for=shift,
                    slot_id=shift.slot_id,
                    empty=shift.empty,
                    is_primary=shift.is_primary,
                )
            )
            # стартом становится конец сформированной смены
            start = start_range

        # добавляем замену с противоположным значением типа занятости empty
        subshift = add_one_subshift(
            time_range=time_range, start=start,
            end=shift.end, shift=shift,
            empty=not shift.empty,
        )
        result.append(subshift)

        start = subshift.end

    if start < shift.end:
        # в конце смены нужно додежурить (в случае, если переданы holidays)
        # или пустой, если переданы внезапные рабочие
        result.append(
            Shift(
                start=start,
                end=shift.end,
                schedule_id=shift.schedule_id,
                replacement_for=shift,
                slot_id=shift.slot_id,
                empty=shift.empty,
                is_primary=shift.is_primary,
            )
        )

    return result


def set_shift_empty(db_obj: Shift, empty: bool) -> None:
    if empty is None or db_obj.empty == empty:
        return

    db_obj.empty = empty

    if empty:
        db_obj.staff_id = None


def set_shift_approved(db_obj: Shift, approved: Optional[bool], author_id: int) -> None:
    if approved is None or db_obj.approved == approved:
        return

    db_obj.approved = approved

    changed_by_id_attr = 'approved_by_id' if approved else 'approved_removed_by_id'
    changed_time_attr = 'approved_at' if approved else 'approved_removed_at'

    setattr(db_obj, changed_by_id_attr, author_id)
    setattr(db_obj, changed_time_attr, now())
    logger.info(f'Set approved: {approved} for shift: {db_obj.id} by {author_id}')


def bulk_set_shift_approved(db: Session, db_objs: Iterable[Shift], approved: Optional[bool], author_id: int) -> None:
    if approved is None or all((shift.approved == approved for shift in db_objs)):
        return

    changed_by_id_attr = 'approved_by_id' if approved else 'approved_removed_by_id'
    changed_time_attr = 'approved_at' if approved else 'approved_removed_at'

    to_update = [shift.id for shift in db_objs if shift.approved != approved]
    if to_update:
        db.query(Shift).filter(Shift.id.in_(to_update)).update({
            Shift.approved: approved,
            getattr(Shift, changed_by_id_attr): author_id,
            getattr(Shift, changed_time_attr): now()
        }, synchronize_session=False)


def set_shift_staff(db: Session, db_obj: Shift, staff_id: Optional[int], recalculate_rating: bool = False) -> None:
    """
    Если передан recalculate_rating, то сразу подсчитает новые рейтинги для следующих шифтов
    """
    from watcher.tasks.shift import start_shift

    prev_staff = db_obj.staff
    if prev_staff and prev_staff.id == staff_id:
        return

    if recalculate_rating:
        rating_difference = get_shift_rating_differences(db, shift=db_obj, new_staff_id=staff_id)
        update_staff_ratings_for_further_shifts(db, db_obj, rating_difference)

    if db_obj.end < now():
        # обновляют смену которая уже завершилась
        # нужно в базе рейтинги обновить
        update_ratings_for_past_shift(db=db, shift=db_obj, new_staff_id=staff_id)
    if db_obj.id and db_obj.status == enums.ShiftStatus.active:
        # отзовём сразу, выдадим, когда все сохранится
        check_and_deprive_member(session=db, shift=db_obj)
        db_obj.status = enums.ShiftStatus.scheduled
        start_shift.delay(shift_id=db_obj.id)

    db_obj.staff_id = staff_id


def get_shift_rating_differences(db: Session, shift: Shift, new_staff_id: Optional[int]) -> Dict[str, float]:
    prev_staff = shift.staff
    if prev_staff and prev_staff.id == new_staff_id:
        return {}

    rating_difference = {}

    if new_staff_id:
        staff = get_staff_by_id(db, new_staff_id)
        if staff:
            rating_difference[staff.login] = shift_total_points(shift)

    if prev_staff:
        rating_difference[prev_staff.login] = -shift_total_points(shift)

    return rating_difference


def shift_total_points(shift: Shift):
    hours = (shift.end - shift.start).total_seconds() / 3600
    points_per_hour = shift.slot.points_per_hour if shift.slot else 1
    return round(hours * float(points_per_hour), 1)


def is_subshift(shift: Shift):
    return shift.replacement_for_id is not None


def need_find_duty(shift: Shift) -> bool:
    """
    Если дежурный для смены уже найден -> False
    """
    return not (
        shift.staff
        and (localize(shift.end) <= now() or shift.approved)
    )


def generate_new_sequence_by_shift(db: Session, shift: Shift) -> Shift:
    """
    Генерирует и сохраняет новую последовательность начиная с конкретной смены
    :return: следующую смену после заданной
    """
    from watcher.tasks.generating_shifts import sequence_shifts

    shifts_intervals = (
        query_shifts_with_intervals_by_schedule(db, shift.schedule_id)
        .filter(Shift.start >= shift.start)
    ).all()

    new_sequence_shifts = [item[0] for item in shifts_intervals]
    new_sequence_interval = set([item[1] for item in shifts_intervals])

    new_sequence = sequence_shifts(db, shift.schedule, new_sequence_shifts, new_sequence_interval)

    need_shift = None
    for i, seq_shift in enumerate(new_sequence):
        if i + 1 < len(new_sequence):
            seq_shift.next = new_sequence[i + 1]
            if seq_shift.id == shift.id:
                need_shift = seq_shift.next

    return need_shift


def get_prev_and_next_shifts(main_shift: Shift, parallel_slot_shifts: list[Shift]) -> Tuple[Shift, Shift]:
    """ Возвращаем текущие prev и next шифты """
    if not main_shift.sub_shifts:
        return main_shift.prev, main_shift.next

    prev_shift = None
    next_shift = None

    # считаем что в рамках параллельных слотов одного интервала,
    # должна быть только одна внешнаяя к интервалу prev и next ссылка
    all_parallel_shifts = main_shift.sub_shifts + parallel_slot_shifts
    all_shifts_set = set(all_parallel_shifts)

    for shift in all_parallel_shifts:
        # TODO: если prev_shift уже был и мы нашли еще одну - ошибка
        if shift.prev and not prev_shift and shift.prev not in all_shifts_set:
            prev_shift = shift.prev
        if shift.next and not next_shift and shift.next not in all_shifts_set:
            next_shift = shift.next

    return prev_shift, next_shift


def is_full_shift(main_shift: Shift, sub_shifts: List[ShiftPutSchema]) -> bool:
    if len(sub_shifts) > 1:
        return False

    sub_shift = sub_shifts[0]

    return sub_shift.start == main_shift.start and sub_shift.end == main_shift.end


def update_ratings_for_subshifts(prev_shift: Shift, sub_shifts: List[Shift], staff_map: dict) -> None:
    """ Обновляем рейтинги для новых сабшифтов. Берем рейтинги предыдущего шифта и итеративно подсчитываем """
    if not prev_shift:
        return

    if not prev_shift.predicted_ratings:
        # TODO: нужно починить
        # - если нет нужных людей в предыдущем шифте
        # - если нет вообще рейтингов
        return

    predicted_ratings = get_predicted_ratings_after_shift(prev_shift)

    for shift in sub_shifts:
        shift.predicted_ratings = copy.copy(predicted_ratings)

        if shift.staff_id:
            staff = staff_map[shift.staff_id]
            if staff.login in shift.predicted_ratings:
                # а если не в shift.predicted_ratings учесть может нужно такого?
                predicted_ratings[staff.login] += shift_total_points(shift)


def find_other_subshifts(shift: Shift, shifts_sequence: List[Shift]) -> List[Shift]:
    result = []
    for other_shift in shifts_sequence:
        if (
            other_shift.replacement_for_id == shift.replacement_for_id
            and not other_shift.empty
            and need_find_duty(shift=other_shift)
            and other_shift.id != shift.id
        ):
            result.append(other_shift)
    return result
