import logging

from fastapi import Request
from hashlib import sha256
from asgi_yauth.backends.tvm2 import AuthTypes
from asgi_yauth.backends.base import BaseBackend
from asgi_yauth.user import YandexUser
from ylog.context import put_to_context

from watcher.logic.exceptions import (
    NoValidAuthProvided,
    InvalidSecretToken,
)
from watcher.logic.exception_handler import base_exception_handler
from watcher.logic.auth import no_auth_required
from watcher.config import settings
from watcher.logic.logger.base import request_log_context

logger = logging.getLogger(__name__)


async def x_secret_token(request: Request, call_next):
    if settings.ENV_TYPE == settings.ENV_TYPE_PRODUCTION:
        return await call_next(request)

    secret_token = request.headers.get('x-secret-token')
    user_uid = request.headers.get('x-uid')

    if not (secret_token and user_uid):
        return await call_next(request)

    is_valid = sha256(secret_token.encode()).hexdigest() == settings.X_SECRET_HASH
    if not is_valid:
        logger.warning(f'Invalid x-secret-token provided, x-uid: {user_uid}')
        return await base_exception_handler(
            request=request,
            exc=InvalidSecretToken(),
        )

    request.scope['user'] = YandexUser(
        uid=user_uid,
        auth_type='secret_token',
        backend=BaseBackend
    )
    logger.info(f'Successfully authorize by x-secret-token, x-uid: {user_uid}')

    return await call_next(request)


async def yauth_require(request: Request, call_next):
    """
    Возвращаем 401 если пользователь не авторизован
    """
    if request.user.is_authenticated() or no_auth_required(request):
        return await call_next(request)

    return await base_exception_handler(
        request=request,
        exc=NoValidAuthProvided(reason=request.user.reason),
    )


async def log_context(request: Request, call_next):
    if no_auth_required(request):
        response = await call_next(request)
    else:
        with request_log_context(request):
            response = await call_next(request)
            put_to_context('profiling', {
                'status_code': response.status_code
            })
    return response


async def set_tvm_user(request: Request, call_next):
    """
    Устанавливаем пользователя, если пришли с сервис тикетом и заголовком X-UID
    """
    request.scope['service_ticket_only'] = False

    if request.user.is_authenticated() and request.user.auth_type == AuthTypes.service.value:
        # пришли с сервисным тикетом, нужно проверить есть ли uid в заголовке
        # и есть ли права у потребителя приходить от имени пользователей
        user_uid = request.headers.get('x-uid')
        if user_uid and request.user.service_ticket.src in settings.ALLOWED_TVM_CLIENT_FOR_UID_HEADER:
            request.user.uid = user_uid
        else:
            request.scope['service_ticket_only'] = True

    return await call_next(request)
