from sqlalchemy.orm import Session, joinedload
from watcher.db import (
    BotUser,
)
from watcher.logic.clients.staff_api import staff_api_client
from watcher.logic.clients.jns import jns_client
from watcher.tasks.base import lock_task
from watcher.db.base import dbconnect
from watcher.logic.common import get_chunks

STAFF_PAGE_LIMIT = 100


@lock_task(save_metrics=True, send_to_unistat=True)
@dbconnect
def remove_unauthorized_bot_users(session: Session):
    """
    Ежедневно проверяет всех бот юзеров на то что:
    1) Стафф не уволен
    2) Телеграм логин бот юзера не совпадает с телеграмом, привязанным к стаффу
    В случае непрохождения проверки бот юзер удаляется из базы и из jns_client
    """
    bot_users = session.query(BotUser).options(joinedload(BotUser.staff)).all()
    bot_users_map = {}
    staff_telegrams = []
    to_delete = set()
    for user in bot_users:
        if user.staff.is_dismissed:
            to_delete.add(user.username)
        else:
            bot_users_map[user.staff.login] = user.username
    for chunk in get_chunks(list(bot_users_map.keys()), size=STAFF_PAGE_LIMIT):
        staff_telegrams.extend(staff_api_client.get_telegrams_by_logins(chunk, limit=STAFF_PAGE_LIMIT))

    for staff_user in staff_telegrams:
        staff_user_telegrams = set(account['value'] for account in staff_user['telegram_accounts'])
        bot_user_telegram = bot_users_map[staff_user['login']]
        if bot_user_telegram not in staff_user_telegrams:
            to_delete.add(bot_user_telegram)
    if to_delete:
        session.query(BotUser).filter(BotUser.username.in_(to_delete)).delete(synchronize_session=False)
        jns_client.delete_users(list(to_delete))
