import datetime
import logging

from collections import defaultdict, OrderedDict
from sqlalchemy.orm import joinedload, Session
from typing import Dict, Iterable, List, Optional, Tuple, Union
from dateutil.parser import parse

from watcher import enums
from watcher.crud.base import get_object_by_model_or_404
from watcher.crud.holiday import get_holidays
from watcher.crud.interval import query_intervals_with_slots_by_revision, query_interval_by_schedules
from watcher.crud.revision import get_current_revision, query_active_revisions, query_revisions_for_period
from watcher.crud.schedule import (
    get_schedule_or_404,
    query_active_schedules_with_active_revisions,
    query_schedules_by_group,
    save_schedule_shifts_boundary_recalculation_error,
    schedule_has_shifts,
)
from watcher.crud.shift import (
    get_first_shift_by_schedule,
    get_last_shift_by_schedule,
    query_all_shifts_by_schedule,
)
from watcher.config import settings
from watcher.db import Interval, Revision, Schedule, Shift, Slot, Holiday
from watcher.logic.interval import is_empty
from watcher.logic.member import check_and_deprive_member
from watcher.logic.exceptions import InvalidData, RecordNotFound
from watcher.logic.holidays import (
    get_unexpected_workday,
    get_extend_weekends,
    get_unexpected_holidays
)
from watcher.logic.shift import (
    add_subshifts,
    bind_sequence_shifts,
    repair_sequence_shifts,
    generate_new_sequence_by_shift,
)
from watcher.logic.slot import priority
from watcher.logic.timezone import localize, now, today
from watcher.tasks.base import save_schedule_error
from watcher.db.base import dbconnect
from .base import lock_task

logger = logging.getLogger(__name__)


def get_cycle_duration(revision: Revision) -> datetime.timedelta:
    duration = datetime.timedelta()

    for interval in revision.intervals:
        duration += interval.duration

    return duration


def get_interval_cycle(
    session: Session, schedule_id: int, revision: Optional[Revision] = None
) -> OrderedDict[Interval: Iterable[Slot]]:
    """
    Определяем цикл интервалов

    :param session:
    :param schedule_id: id графика
    :param revision: ревизия, относительно которой, нужно построить цикл.
        Если нужно построить текущий цикл, то ревизия должна быть в статусе active.
        Если ревизия не передана, пытаемся найти активную ревизию.
    :return: возвращает список интервалов и их слотов, например,
        [(interval_1, (slot_id_1, slot_id_2, slot_id_3)), (interval_2, (slot_id_4,)), (interval_3, (slot_id_6, )), ...]
    """

    if revision is None:
        revision = get_current_revision(db=session, schedule_id=schedule_id)

    if revision.schedule_id != schedule_id:
        raise InvalidData(message={
            'ru': 'Revision не соответствует Schedule',
            'en': 'Revision does not match Schedule',
        })
    interval_cycle = OrderedDict()
    query = query_intervals_with_slots_by_revision(db=session, revision_id=revision.id)
    for interval, slot in query.all():
        interval_cycle.setdefault(interval, [])
        if slot:
            interval_cycle[interval].append(slot)

    return interval_cycle


def make_duplicates_shift(slots: Iterable[Slot], shift: Shift, replacement_shifts: Iterable[Shift]) -> List[Shift]:
    """Формируем дубликаты шифта и его замен, если слотов в интервале больше одного"""

    result = []
    if len(slots) == 1:
        return result

    for slot in slots[1:]:
        # нулевой слот - это уже сформированный шифт
        new_shift = shift.copy(slot_id=slot.id, is_primary=slot.is_primary)
        result.append(new_shift)
        for rep_shift in replacement_shifts:
            new_rep = rep_shift.copy(slot_id=slot.id, replacement_for=new_shift, is_primary=slot.is_primary)
            result.append(new_rep)

    return result


def get_cutted_days(
    session: Session, shift: Shift,
    holidays: Iterable[Holiday], threshold: datetime.datetime,
    cut_days_by_threshold: bool = False,
    extend_weekends: bool = False, unexpected_holidays: bool = False,
) -> (List[datetime.date], int):
    """
    Находим все дни, которые нужно вырезать из смены
    :param cut_days_by_threshold: флаг пропуска дней больше threshold
    :return:
        days_to_cut - список дней которые нужно вырезать,
        shift_extension - число дней, на которое нужно продлить смену
    """
    days_to_cut = []
    shift_extension = 0
    start = shift.start
    end = shift.end
    while True:
        extend_weekends_days = None
        if extend_weekends:
            extend_weekends_days = get_extend_weekends(start, end)
            shift_extension += len(extend_weekends_days)
            days_to_cut.extend(extend_weekends_days)
        if unexpected_holidays:
            unexpected_holidays_dates = get_unexpected_holidays(
                session=session, start=start,
                end=end, holidays=holidays,
            )
            days_to_cut.extend(h.date for h in unexpected_holidays_dates)
        if extend_weekends_days:
            # последний день уже проверен, кроме случая конца смены в 00:00
            start = end if end.time() == datetime.time(0, 0) else end + datetime.timedelta(days=1)
            end += datetime.timedelta(days=len(extend_weekends_days))
            if cut_days_by_threshold:
                end = min(end, threshold)
        else:
            return days_to_cut, shift_extension


def create_shifts(
    session: Session, start: datetime.datetime,
    interval_cycle: OrderedDict[Interval: Iterable[Slot]],
    schedule_id: int,
    holidays: Iterable[Holiday], threshold: datetime.datetime,
    cut_shifts_by_threshold: bool = False,
) -> Tuple[Shift, List[Shift]]:
    """
    Формирование шифтов

    :param start:
        - стартовая точка для формируемого цикла шифтов
    :param interval_cycle:
        - цикл в виде упорядоченного словаря интервалов с принадлежащими им слотами
    :param schedule_id:
    :param holidays: список выходных/праздников от старта до threshold`а графика
    :param threshold: конец ревизии, для того чтобы не заехать на следующую
    :param cut_shifts_by_threshold: флаг обрезки шифтов по трешхолду
    :param session:
    :return:
        - последний сформированный шифт и список всех шифтов
        - все шифты сформированы, но ещё не добавлены в базу
    """

    expected_count = 0
    last_shift = None
    created_shifts = []

    for interval, slots in interval_cycle.items():
        if is_empty(interval) and not slots:
            logger.warning("Для пустого интервала не было слотов. Идет создание...")
            # для пустого интервала при сохранении должен создаваться слот без состава
            slot = Slot(interval=interval)
            session.add(slot)
            slots.append(slot)
        elif not is_empty(interval) and not slots:
            save_schedule_shifts_boundary_recalculation_error(
                db=session,
                schedule_id=schedule_id,
                message=f"У непустого интервала (id={interval.id}) нет слотов",
                raise_exception=InvalidData
            )

        expected_count += len(slots)
        end = start + interval.duration
        # формируем шифт
        shift = Shift(
            start=localize(start),
            end=localize(end),
            schedule_id=schedule_id,
            empty=is_empty(interval),
        )
        replacement_shifts = []
        shift.slot_id = slots[0].id
        shift.is_primary = slots[0].is_primary

        cutted_days, shift_extension = get_cutted_days(
            session=session,
            shift=shift,
            holidays=holidays,
            extend_weekends=interval.weekend_behaviour == enums.IntervalWeekendsBehaviour.extend,
            unexpected_holidays=interval.unexpected_holidays == enums.IntervalUnexpectedHolidays.remove,
            threshold=threshold,
            cut_days_by_threshold=cut_shifts_by_threshold,
        )
        if shift_extension:
            # продляем смену в случае вырезанных выходных, но не дальше трешхолда
            # если cut_shifts_by_threshold = True
            shift.end += datetime.timedelta(days=shift_extension)
            if cut_shifts_by_threshold:
                shift.end = min(shift.end, threshold)
        if not is_empty(interval):
            # если это не пустой шифт, то нам нужно обработать вырезанные дни
            # для этого создаем пустую подсмену
            if cutted_days:
                replacement_shifts = add_subshifts(cutted_days, shift)
        else:
            # а если шифт относится к пустому интервалу, то нужно правильно обработать "внезапный" рабочий
            # сейчас по умолчанию создаём непустую подсмену
            unexpected_workday = get_unexpected_workday(
                session=session, start=shift.start, end=shift.end,
                holidays=holidays,
            )
            if unexpected_workday:
                # формируем замены
                replacement_shifts = add_subshifts(cut_days=unexpected_workday, shift=shift)

        created_shifts.append(shift)
        created_shifts.extend(replacement_shifts)

        # проверим и создадим дубликаты, если слотов у интервала больше 1
        duplicates_shifts = make_duplicates_shift(slots=slots, shift=shift, replacement_shifts=replacement_shifts)
        created_shifts.extend(duplicates_shifts)

        start = shift.end
        last_shift = shift

    # если число основных шифтов (не подсмен) не совпадает с количеством слотов у интервалов,
    # то шифты где-то задваиваются или недоформировываются,
    # такой набор смен сохранять нельзя
    result_count = len([shift for shift in created_shifts if shift.replacement_for is None])
    if result_count != expected_count:
        raise InvalidData(
            message={
                'ru': (f'Сформировано неверное число основных шифтов внутри одного цикла. '
                       f'Ожидалось: {expected_count}, сформировано: {result_count}'),
                'en': (f'An incorrect number of basic shifts was formed within one cycle. '
                       f'Expected: {expected_count}, formed: {result_count}'),
            }

        )

    return last_shift, created_shifts


def priority_of_slots(
    session: Session,
    schedule: Schedule,
    interval_ids: List[Interval],
    for_group: bool = False
) -> Dict[int, float]:
    """
    Приоритезируем слоты. Возможны режимы: внутри графика или внутри группы

    :param session: текущая сессия
    :param schedule: график
    :param interval_ids: id интервалов
    :param for_group: режим приоритезации, внутри группы или графика

    :return: словарь из id слотов, в качестве значений - их приоритеты
    """

    schedule_ids = [schedule.id, ]
    if for_group:
        schedule_ids = query_schedules_by_group(
            db=session,
            schedules_group_id=schedule.schedules_group_id,
            query=session.query(Schedule.id)
        ).all()

    if interval_ids is None:
        interval_ids = query_interval_by_schedules(
            db=session,
            schedule_ids=schedule_ids,
            query=session.query(Interval.id)
        ).all()

    all_slots = (
        session.query(Slot)
            .options(
            joinedload('composition'),
            joinedload('composition.participants'),
            joinedload('interval'),
        )
            .filter(Slot.interval_id.in_(interval_ids)).all()
    )

    result_dict = {slot.id: priority(slot.composition, slot.interval) for slot in all_slots}

    return result_dict


def sorted_shifts_for_group(shifts):
    """
    Внутри группы дубликаты слотов могут идти только после того,
    как слоты других графиков на эту дату уже есть в приоритезированном списке.
    Такое условие необходимо, чтобы мы могли дать больше гарантий,
    что хотя бы на один из слотов каждого из графиков будет хоть кто-то назначен.
    """

    sorted_result = []
    shift_prev = None
    duplicate_shifts = []
    schedules_ids = set()
    for shift in shifts:
        if shift_prev is not None:
            if shift.start == shift_prev.start:
                if shift.schedule_id in schedules_ids:
                    duplicate_shifts.append(shift)
                    continue

            else:
                schedules_ids = set()
                sorted_result.extend(duplicate_shifts)
                duplicate_shifts = []
        sorted_result.append(shift)
        schedules_ids.add(shift.schedule_id)
        shift_prev = shift
    sorted_result.extend(duplicate_shifts)
    return sorted_result


def sequence_shifts(
    session: Session, schedule: Schedule, shifts: List[Shift], intervals: Iterable[Interval], for_group: bool = False
) -> List[Shift]:
    """
    Формируем последовательность шифтов

    :param schedule: график
    :param shifts: уже отсортированные по времени начала шифты
    :param intervals: интервалы шифтов из разных ревизий
    :param for_group: признак внутри группы или внутри графика нужно произвести рассчет последовательности
        - внутри графика рассчитываем и записываем при создании шифтов
        - внутри группы нужно для распредлеения людей, запись не должна происходить

    :return: возвращаем отсортированный список, это и будет последовательность,
        которую нужно будет дополнительно сохранить в поле next у шифта
        (когда формируем последовательность внутри графика)
        - В данном методе сохранения не происходит, это вынесено за его рамки и должно происходить в основном методе
    """
    # Собираем словарь смен вида {datetime: [шифты]}
    start_to_shift = defaultdict(list)
    shifts_to_sequence = []
    main_shifts = set()

    for shift in shifts:
        # TODO: в таске про запросы ABC-11059 тут нужно обязательно проверить,
        #   что если у нас существующие шифты, то мы не делаем доп запросы
        if shift in main_shifts or shift.replacement_for in main_shifts:
            continue

        if shift.sub_shifts:
            # если мы не встречали основной шифт, то добавляем все его подшифты
            shifts_to_sequence.extend(sorted(shift.sub_shifts, key=lambda x: x.start))
        elif not shift.replacement_for:
            # если у шифта нет подшифтов и у он не является подшифтов, то просто добавляем в список
            shifts_to_sequence.append(shift)
        elif shift.replacement_for not in main_shifts:
            # если основная смены подсмены еще не встречалась, то добавим все подсмены которые начинаются позже текущей
            start_date = shift.start
            subshifts = sorted(shift.replacement_for.sub_shifts, key=lambda x: x.start)

            for sub_shift in subshifts:
                if sub_shift.start >= start_date:
                    shifts_to_sequence.append(sub_shift)
                    start_to_shift[localize(sub_shift.start)].append(sub_shift)

            shift = shift.replacement_for

        main_shifts.add(shift)
        start_to_shift[localize(shift.start)].append(shift)

    # Если на одно время нет по несколько смен,
    # то можем просто вернуть то, что пришло, тк на входе ожидаем уже отсортированный список
    multiple_shifts = False
    for start_shifts in start_to_shift.values():
        if len(start_shifts) > 1:
            multiple_shifts = True
            break

    if for_group:
        slots_with_priority = priority_of_slots(
            session=session,
            schedule=schedule,
            interval_ids=[interval.id for interval in intervals],
            for_group=for_group,
        )
        result = sorted(
            shifts_to_sequence,
            key=lambda x: (x.start, -(slots_with_priority[x.slot_id]), x.slot_id)
        )

        return sorted_shifts_for_group(shifts=result)

    if not multiple_shifts:
        return shifts_to_sequence

    revision_intervals = defaultdict(list)
    revision_slot_ids = defaultdict(set)
    slot_id_revisions = {}
    for interval in intervals:
        revision = interval.revision
        revision_intervals[revision].append(interval)
        for slot in interval.slots:
            revision_slot_ids[revision].add(slot.id)
            slot_id_revisions[slot.id] = revision

    revision_shifts = defaultdict(list)
    for shift in shifts_to_sequence:
        revision = slot_id_revisions[shift.slot_id]
        revision_shifts[revision].append(shift)

    result = []
    for revision in sorted(revision_intervals, key=lambda x: x.apply_datetime):
        current_revision_intervals = revision_intervals[revision]
        interval_ids = [interval.id for interval in current_revision_intervals]

        slots_with_priority = priority_of_slots(
            session=session,
            schedule=schedule,
            interval_ids=interval_ids,
            for_group=for_group
        )
        result.extend(
            sorted(
                revision_shifts[revision],
                key=lambda x: (x.start, -(slots_with_priority[x.slot_id]), x.slot_id)
            )
        )

    return result


def create_shifts_by_revision(
    session: Session, start: datetime, threshold: datetime,
    interval_cycle: OrderedDict[Interval: Iterable[Slot]], schedule: Schedule,
    revision: Revision, prev_shift: Optional[Shift] = None, force: Optional[int] = None,
) -> tuple[Shift, list[Shift], int]:
    """
    Строим по шифты по конкретной ревизии

    :param session
    :param start
    :param threshold
    :param interval_cycle
    :param schedule
    :param revision
    :param prev_shift: шифт, который последний на данный момент.
        Он должен в итоге ссылаться на первы, что тут создаётся
    :param force: принудительное количество циклов
    """
    calculated_cycles = 0
    shifts = []
    last_shift = prev_shift
    holidays = get_holidays(db=session, start=start, end=threshold).all()
    # шифты создаём на указанное в настройках графика кол-во дней,
    # или на строго заданное кол-во циклов
    while start < threshold or (force is not None and calculated_cycles < force):
        last_shift, new_shifts = create_shifts(
            session=session,
            start=start,
            interval_cycle=interval_cycle,
            schedule_id=schedule.id,
            holidays=holidays,
            threshold=threshold,
            cut_shifts_by_threshold=bool(revision.next),
        )

        shifts.extend(new_shifts)
        start = localize(last_shift.end)
        calculated_cycles += 1

    if shifts:
        shifts_sorted_by_priority = sequence_shifts(
            session=session,
            schedule=schedule,
            shifts=shifts,
            intervals=interval_cycle.keys()
        )

        # сохраним полученную последовательность
        bind_sequence_shifts(shifts_sorted_by_priority)
        if prev_shift:
            shifts_sorted_by_priority[0].prev = prev_shift

    return last_shift, shifts, calculated_cycles


@lock_task(lock_key=lambda schedule_id, *args, **kwargs: 'initial_creation_of_shifts_{}'.format(schedule_id))
@dbconnect
@save_schedule_error
def initial_creation_of_shifts(schedule_id: int, session: Session = None):
    """
    Первичное создание шифтов

    :param
        * id графика, для которого нужно создать шифты

    Запукается:
        * только из ручки создания графика.

    :return:
    """
    logger.info(f'Processing initial_creation_of_shifts for schedule: {schedule_id}')
    from watcher.tasks.people_allocation import start_people_allocation

    if schedule_id is None:
        return

    schedule = get_schedule_or_404(session, schedule_id)

    # если у графика уже есть шифты, то создавать их не нужно
    if schedule_has_shifts(session, schedule_id):
        logger.info(f'Schedule has shifts, skipping {schedule_id}')
        return

    # строим цикл интервалов по текущей ревизии
    # TODO: теоретически тут тоже уже может быть несколько ревизий
    try:
        revision = get_current_revision(db=session, schedule_id=schedule_id)
    except RecordNotFound:
        revision = query_active_revisions(
            db=session, schedule_id=schedule_id
        ).order_by(Revision.apply_datetime).first()
        if not revision:
            logger.error(f'No revision for schedule: {schedule_id}, skipping')
            return
    interval_cycle = get_interval_cycle(session=session, schedule_id=schedule_id, revision=revision)

    start = localize(revision.apply_datetime)
    threshold = localize(datetime.datetime.combine(today(), datetime.datetime.min.time()) + schedule.threshold_day)
    last_shift = None
    # если мы дошли по трешхолда, но циклов недостаточно, нужно запустить принудительно
    last_shift, shifts, new_calculated_cycles = create_shifts_by_revision(
        session=session,
        start=start,
        threshold=threshold,
        interval_cycle=interval_cycle,
        schedule=schedule,
        revision=revision,
        prev_shift=last_shift,
        force=settings.MIN_COUNT_OF_CYCLES
    )

    session.add_all(shifts)

    # Коммит для того чтобы шифты в тестах обновились до запуска start_people_allocation
    session.commit()

    # ищем дату, с которой нужно перераспределять людей
    start_date = localize(revision.apply_datetime)
    if start_date < now():
        for past_shift in shifts:
            if localize(past_shift.start) <= now() <= localize(past_shift.end):
                start_date = localize(past_shift.start)
                break

    start_people_allocation(
        session=session,
        schedules_group_id=schedule.schedules_group_id,
        start_date=start_date,
    )


def get_actual_revisions(
    db: Session,
    global_threshold: datetime,
    date_from_create: datetime,
    schedule_id: int,
    past_revision: Optional[Revision] = None,
    current_revision: Optional[Revision] = None,
) -> Tuple[Iterable[Revision], Revision]:
    """
    Возвращаем список актуальных ревизий, по которым нужно произвести построение, а также ревизию последнего шифта
    """
    # выбираем все ревизии в промежутке c даты последней смены до global_threshold'a
    revisions = query_active_revisions(
        db=db,
        schedule_id=schedule_id,
        query=db.query(Revision).filter(
            Revision.apply_datetime < global_threshold,
            Revision.apply_datetime > date_from_create
        )
    ).order_by(Revision.apply_datetime).all()

    actual_revisions = {}
    last_shift_revision = past_revision
    if not revisions or localize(revisions[0].apply_datetime) > date_from_create:
        # значит будущих нет или в момент старта не начинается новая ревизия,
        # нужно взять ещё и текущую ревизию для последнего созданного шифта
        # здесь и далее current_revision - это текущая ревизия относительно последнего шифта (date_from_create)
        if not last_shift_revision:
            if current_revision:
                last_shift_revision = current_revision
            else:
                # но здесь не выбираем именно как ревизию последнего шифта
                # чтобы сделать далее вывод о том, всё ли корректно
                last_shift_revision = get_current_revision(db=db, schedule_id=schedule_id)

        actual_revisions[localize(date_from_create)] = last_shift_revision

    for revision in revisions:
        actual_revisions[localize(revision.apply_datetime)] = revision

    return actual_revisions, last_shift_revision


def get_revisions(
    db: Session, schedule_id: int, date_from_create: datetime,
    first_shift: Shift, global_threshold: datetime,
) -> dict:
    past_revisions = query_revisions_for_period(
        db=db,
        schedule_id=schedule_id,
        end=date_from_create,
        start=first_shift.slot.interval.revision.apply_datetime,
    ).all()
    past_revision = None if not past_revisions else past_revisions[-1]
    revisions, _ = get_actual_revisions(
        db=db,
        global_threshold=global_threshold,
        date_from_create=date_from_create,
        schedule_id=schedule_id,
        past_revision=past_revision,
        current_revision=None,
    )

    return revisions


def complete_cycle(
    db: Session, interval_cycle: OrderedDict[Interval: Iterable[Slot]], last_shift: Shift,
    schedule: Schedule, revision: Revision, threshold: datetime, global_threshold: datetime
) -> list[Shift]:
    """
    Достраиваем цикл до полного
    """
    interval_cycle_itr = iter(interval_cycle.items())
    slot_found = False

    for _, last_slots in interval_cycle_itr:
        if last_shift.slot in last_slots:
            slot_found = True
            break

    if not slot_found:
        return []

    # Формируем новый цикл интервалов, которые не были задействованы в прошлом цикле
    rest_interval_cycle = OrderedDict()
    for key, val in interval_cycle_itr:
        rest_interval_cycle[key] = val

    # Формируем шифты для оставшихся интервалов цикла
    last_shift, new_shifts, calculated_cycles = create_shifts_by_revision(
        session=db,
        start=localize(last_shift.end),
        threshold=threshold,
        interval_cycle=rest_interval_cycle,
        schedule=schedule,
        revision=revision,
        prev_shift=last_shift,
        force=1,
    )

    return reducing_shifts(
        threshold=threshold,
        global_threshold=global_threshold,
        new_shifts=new_shifts,
    )


def create_dry_shifts(
    db: Session, global_threshold: datetime, revisions: Iterable[Revision], schedule: Schedule,
    last_shift: Shift, last_shift_revision: Revision, boundaries_start_date: datetime, cycle_count: Optional[int] = 0
) -> List[Shift]:
    """
    Строим представление шифтов, именно то, что должны получить.
    """
    dry_shifts = []
    for start, revision in revisions.items():
        threshold = global_threshold
        if revision.next:
            threshold = min(localize(revision.next.apply_datetime), global_threshold)

        interval_cycle = get_interval_cycle(session=db, schedule_id=schedule.id, revision=revision)

        if revision == last_shift_revision and localize(last_shift.end) != localize(threshold):
            last_slots = next(reversed(interval_cycle.values()))
            if last_shift.slot not in last_slots:
                # Если последний шифт не входит в последний интервал цикла (случилось что-то непредвиденное)
                # видимо нужно достроить, но если это конец по трешхолду, то все в порядке
                # ситуация, что интервал вообще не из этой ревизии - обработана выше
                cycle_shifts = complete_cycle(
                    db=db,
                    interval_cycle=interval_cycle,
                    last_shift=last_shift,
                    schedule=schedule,
                    revision=revision,
                    threshold=threshold,
                    global_threshold=global_threshold,
                )

                if cycle_shifts is None:
                    logger.info(
                        f'Sending revision_shift_boundaries for schedule: {schedule.id},'
                        f' from: {boundaries_start_date}, no cycle_shifts'
                    )
                    revision_shift_boundaries.delay(
                        schedule_id=schedule.id,
                        date_from=boundaries_start_date,
                    )
                    return []

                dry_shifts.extend(cycle_shifts)
                if cycle_shifts:
                    last_shift = cycle_shifts[-1]
                    continue

        # Формируем полноценные циклы, если необходимо
        last_shift, new_shifts, calculated_cycles = create_shifts_by_revision(
            session=db,
            start=localize(last_shift.end),
            threshold=threshold,
            interval_cycle=interval_cycle,
            schedule=schedule,
            revision=revision,
            prev_shift=last_shift,
        )

        cycle_count += calculated_cycles
        if new_shifts:
            dry_shifts.extend(
                reducing_shifts(
                    threshold=threshold,
                    global_threshold=global_threshold,
                    new_shifts=new_shifts,
                )
            )

    # если циклов не достаточно, достраиваем циклы только последней ревизией
    # если MIN_COUNT_OF_CYCLES не изменится, то зайдём сюда не более 1 раза

    if cycle_count < settings.MIN_COUNT_OF_CYCLES:
        threshold = global_threshold
        force = settings.MIN_COUNT_OF_CYCLES - cycle_count
        if revision.next:
            # если где-то там планируется еще одна ревизия, то выйдем из create_shifts_by_revision
            # по трешхолду или по force (если трешхолд очень далёкий)
            threshold = localize(revision.next.apply_datetime)

        _, new_shifts, _ = create_shifts_by_revision(
            session=db,
            start=localize(last_shift.end),
            threshold=threshold,
            interval_cycle=interval_cycle,
            schedule=schedule,
            revision=revision,
            prev_shift=last_shift,
            force=force,
        )

        if revision.next:
            new_shifts = reducing_shifts(
                threshold=threshold,
                global_threshold=global_threshold,
                new_shifts=new_shifts
            )

        dry_shifts.extend(new_shifts)

    return dry_shifts


def match_shifts(shifts, dry_shifts):
    shifts_to_delete, shifts_to_add, actual_shifts = [], [], []
    new_old_shift_mapping = {}
    complete = False
    i = j = 0
    number_of_continuous_deletions = 0
    while i < len(shifts) and j < len(dry_shifts) and not complete:
        shift = shifts[i]
        new_shift = dry_shifts[j]

        shift.replacement_for = None
        if new_shift.replacement_for in new_old_shift_mapping:
            new_shift.replacement_for = new_old_shift_mapping[new_shift.replacement_for]

        # Флаги перехода к следующим шифтам
        change_shift = False
        change_new_shift = False
        # сравним текущий с представлением
        if localize(shift.start) == localize(new_shift.start) and localize(shift.end) == localize(new_shift.end):
            # new_shift не должен попасть в add, скопируем только нужные поля
            change_shift = True
            change_new_shift = True

            shift.slot_id = new_shift.slot_id
            if new_shift.replacement_for in actual_shifts:
                shift.replacement_for = new_shift.replacement_for
            actual_shifts.append(shift)
            new_old_shift_mapping[new_shift] = shift

            # обнуляем число удаленных подряд смен
            number_of_continuous_deletions = 0
            if shift.id != new_shift.id:
                shift.empty = new_shift.empty
                if new_shift.empty:
                    shift.staff = None

        elif localize(shift.end).date() <= localize(new_shift.start).date():
            # если n смен подряд уже перезаписались,
            #   то тогда нужно все затиреть и перезаписать
            if number_of_continuous_deletions > 2:
                complete = True

            else:
                shifts_to_delete.append(shift)
                change_shift = True
                number_of_continuous_deletions += 1

        elif localize(shift.start).date() >= localize(new_shift.end).date():
            if number_of_continuous_deletions > 2:
                complete = True

            else:
                shifts_to_add.append(new_shift)
                change_new_shift = True
                actual_shifts.append(new_shift)

                # обнуляем число удаленных подряд смен
                number_of_continuous_deletions = 0

        else:  # если есть пересечение
            duration_of_general_part = (
                min(localize(shift.end), localize(new_shift.end)) -
                max(localize(shift.start), localize(new_shift.start))
            )
            # общая продолжительность - от самого раннего старта до самого позднего конца:
            entire_duration = (
                max(localize(shift.end), localize(new_shift.end)) -
                min(localize(shift.start), localize(new_shift.start))
            )
            if (entire_duration - duration_of_general_part) > datetime.timedelta(days=2):
                shifts_to_delete.append(shift)
                number_of_continuous_deletions += 1
                shifts_to_add.append(new_shift)
                actual_shifts.append(new_shift)

            else:
                # старт шифта в прошлом, то не удаляем их, а перезаписываем,
                # даже если пересчений меньше необходимого
                shift.start = new_shift.start
                shift.end = new_shift.end
                shift.empty = new_shift.empty
                shift.slot_id = new_shift.slot_id

                if new_shift.replacement_for in actual_shifts:
                    shift.replacement_for = new_shift.replacement_for

                if new_shift.empty:
                    shift.staff = None

                number_of_continuous_deletions = 0
                actual_shifts.append(shift)
                new_old_shift_mapping[new_shift] = shift

            change_shift = True
            change_new_shift = True

        # досрочно завершим сверку и добавим новые и удалим все старые
        if complete:
            shifts_to_delete.extend(shifts[i:])
            shifts_to_add.extend(dry_shifts[j:])
            actual_shifts.extend(dry_shifts[j:])
            break

        # перейдём к следующему
        if change_shift:
            i += 1

        if change_new_shift:
            j += 1

    else:
        shifts_to_add.extend(dry_shifts[j:])
        actual_shifts.extend(dry_shifts[j:])
        shifts_to_delete.extend(shifts[i:])

    return shifts_to_delete, shifts_to_add, actual_shifts


@lock_task(lock_key=lambda schedule_id, *args, **kwargs: 'revision_shift_boundaries_{}'.format(schedule_id))
@dbconnect
@save_schedule_error
def revision_shift_boundaries(
    session: Session,
    revision_id: Optional[int] = None,
    schedule_id: Optional[int] = None,
    date_from: Optional[Union[datetime.datetime, str]] = None,
    from_holiday: Optional[bool] = False,
):
    """
    Таска пересчёта границ.
    Перестраиваем график по данным ревизий.
    Интересуют только текущая активная ревизия и будущие.

    Существующие шифты перебираем исходя из поля next,
    поэтому при удалении лишних не нужно дополнительно проверять приоритетность слотов.

    date_from - дата, с которой задаётся пересчёт
    Но в по факту пересчет будет с date_from_create - со старта первого шифта,
    который на момент date_from ещё не закончился


    Какие тут кейсы
      - добавили ревизию (передать сюда id, посмотреть на дату старта ревизии,
      обрезать текущие смены которые идут в момент ее начала - все будущие удалить
      запустить proceed_new_shifts)
      - удалили ревизию (передать сюда время старта, и action=delete,
      взять смены start >= этой даты - удалить все запустить proceed_new_shifts)
      - добавился праздник - (
        посмотреть на настройки расписания в момент праздника (если игнорируем ничего не делать)
        посмотреть на шифты - есть в момент праздника шифты то? если нет - ничего не делать
        если есть - удалить все шифты начиная с того, что идет во время праздника
        запустить proceed_new_shifts проверки тут выполняются при изначальном запуске таски
      )
    """
    logger.info(f'Processing revision_shift_boundaries for schedule: {schedule_id}, from: {date_from}, revision: {revision_id}')

    if not date_from:
        date_from = now()
    elif isinstance(date_from, str):
        date_from = parse(date_from)

    schedule = get_schedule_or_404(db=session, schedule_id=schedule_id)
    if not schedule_has_shifts(session, schedule_id):
        logger.info(f'Starting initial_creation_of_shifts for {schedule_id}')
        initial_creation_of_shifts.delay(schedule_id=schedule_id)
        return

    if revision_id:
        revision = get_object_by_model_or_404(db=session, model=Revision, object_id=revision_id)
        schedule_id = revision.schedule_id
        date_from = revision.apply_datetime
        if (
            not session.query(Shift).filter(
                Shift.schedule_id == schedule_id,
                Shift.end > revision.apply_datetime,
            ).count()
        ):
            # не нужно ничего делать, ревизия начинается после последней смены
            logger.info(f'Skipping revision_shift_boundaries for revision {revision_id}')
            return

        if date_from > now() and [rev.id for rev in query_active_revisions(db=session, schedule_id=schedule_id)] == [revision_id]:
            # есть только одна ревизия, нужно удалить все смены, тогда запустится initial_creation_of_shifts
            logger.info('Deleting all shifts in schedule')
            session.query(Shift).filter(
                Shift.schedule_id==schedule.id
            ).delete(synchronize_session='fetch')
            initial_creation_of_shifts.delay(schedule_id=schedule_id)
            return

    # текущие (и будущие) шифты на момент времени date_from
    query = query_all_shifts_by_schedule(
        db=session,
        schedule_id=schedule_id,
        query=session.query(Shift).filter(Shift.end > date_from),
    ).options(
        joinedload(Shift.sub_shifts)
    )

    # первый шифт, с которого нужно считать
    first_shift, valid = get_first_shift_by_schedule(
        db=session,
        schedule_id=schedule_id,
        query=query,
    )
    if not valid:
        logger.warning(
            f'Found multiple first shifts for {schedule_id}, '
            f'revision_id: {revision_id}, date_from={date_from}'
        )
        # починим последовательность и перезапустим
        generate_new_sequence_by_shift(db=session, shift=first_shift)
        revision_shift_boundaries.delay(
            schedule_id=schedule_id,
            date_from=date_from,
            revision_id=revision_id,
            from_holiday=from_holiday,
        )
        return

    if not first_shift:
        # нечего сверять и перестраивать, попробуем просто достроить
        proceed_new_shifts.delay(schedule_id=schedule_id)
        return

    last_shift = get_last_shift_by_schedule(db=session, schedule_id=schedule.id)
    if not last_shift:
        logger.warning('Sequence of shifts found broken in revision_shift_boundaries')
        repair_sequence_shifts(db=session, schedule_id=schedule_id)
        revision_shift_boundaries.delay(
            schedule_id=schedule_id,
            date_from=date_from,
            revision_id=revision_id,
            from_holiday=from_holiday,
        )
        return
    if not from_holiday:
        # обрежем текущие
        to_reduce = query.filter(Shift.start < date_from).all()
        to_reduce += [
            subshift for shift in to_reduce
            for subshift in shift.sub_shifts
            if subshift.start < date_from
        ]
        for shift in to_reduce:
            if shift.end > date_from:
                shift.end = date_from

        # удалим будущие
        to_delete_shifts = query.filter(Shift.start >= date_from).all()
    else:
        # если добавился/удалился праздник - не обрезаем, а просто удаляем все будущие смены
        # они потом достроются
        to_delete_shifts = query.all()

    to_delete = []
    for shift in to_delete_shifts:
        to_delete.append(shift.id)
        to_delete.extend(obj.id for obj in shift.sub_shifts)

    session.query(Shift).filter(
        Shift.next_id.in_(to_delete)
    ).update(
        {Shift.next_id: None},
        synchronize_session=False,
    )

    session.query(Shift).filter(
        Shift.id.in_(to_delete)
    ).delete(synchronize_session=False)
    session.commit()
    # запустим построение
    proceed_new_shifts(session=session, schedule_id=schedule_id)


def reducing_shifts(threshold: datetime, global_threshold: datetime, new_shifts: List[Shift]) -> List[Shift]:
    """
    Обрезание шифта/шифтов
    Необходимо только если высчитанный трешхолд для
    ревизии не соотвествует глобальному (для всего расчёта)
    """
    if threshold != global_threshold and localize(new_shifts[-1].end) > threshold:
        i = 0
        j = len(new_shifts) - 1
        while i < j and localize(new_shifts[i].start) < threshold <= localize(new_shifts[j].end):
            i += 1
            j -= 1

        if i == j:
            new_shifts[i].end = threshold
            new_shifts[i].next = None
            return new_shifts[:i + 1]

        elif localize(new_shifts[i].start) >= threshold:
            new_shifts[i - 1].end = threshold
            new_shifts[i - 1].next = None
            return new_shifts[:i]

        elif localize(new_shifts[j].end) < threshold:
            new_shifts[j + 1].end = threshold
            new_shifts[j + 1].next = None
            return new_shifts[:j + 2]

        else:
            # следующий в списке может быть аналогичным шифтом
            # из паралелльного слота
            # мы нашли крайние шифты с начала и конца, теперь нужно каждый из них урезать
            k = min(i, j)
            stop = max(i, j)
            while k <= stop:
                new_shifts[k].end = threshold
                k += 1

            new_shifts[k - 1].next = None
            return new_shifts[:k]

    return new_shifts


def get_cycle_count(
    db: Session, past_revisions: Iterable[Revision], last_shift_end: datetime, schedule_id: int
) -> Tuple[int, Revision]:
    # каждая ревизия, попавшая в выборку, принесет как минимум 1 цикл
    cycle_count = len(past_revisions)  # в грубом приближении
    current_revision = None
    if cycle_count == 0:
        # TODO: поправить в ABC-10890
        # после ребейза научить передавать квери, если еще не будет, и выгрузить сразу интервалы этой ревизии
        current_revision = get_current_revision(db=db, schedule_id=schedule_id)
        cycle_duration = get_cycle_duration(current_revision)
        cycle_count = (last_shift_end - now()) // cycle_duration

    return cycle_count, current_revision


@lock_task(lock_key=lambda schedule_id, *args, **kwargs: 'proceed_new_shifts_{}'.format(schedule_id))
@dbconnect
@save_schedule_error
def proceed_new_shifts(schedule_id: int, session: Session):
    """
    Продолжение создания шифтов для расписания

    :param
        * id графика, для которого нужно создать шифты

    :return:
    """
    logger.info(f'Processing proceed_new_shifts for schedule: {schedule_id}')

    from watcher.tasks.people_allocation import start_people_allocation

    if schedule_id is None:
        return

    if not schedule_has_shifts(session, schedule_id):
        initial_creation_of_shifts.delay(schedule_id=schedule_id)
        return

    schedule = get_schedule_or_404(session, schedule_id)
    last_shift = get_last_shift_by_schedule(session, schedule.id)

    if not last_shift:
        logger.warning('Последовательность шифтов была сломана в таске proceed_new_shifts')
        repair_sequence_shifts(session, schedule_id)
        proceed_new_shifts.delay(schedule_id=schedule_id)
        return

    date_from_create = localize(last_shift.start)

    # нужно определить сколько уже циклов построено с текущего момента до последнего шифта
    # количество ревизий, который начнут в этот срок - это количество смен ревизий,
    past_revisions = query_revisions_for_period(
        db=session, schedule_id=schedule_id,
        end=date_from_create
    ).all()
    cycle_count, current_revision = get_cycle_count(
        db=session,
        past_revisions=past_revisions,
        last_shift_end=localize(last_shift.end),
        schedule_id=schedule_id,
    )

    # ниже код непосредственно про достройку шифтов
    past_revision = None if not past_revisions else past_revisions[-1]
    global_threshold = localize(
        datetime.datetime.combine(today(), datetime.datetime.min.time())
        + schedule.threshold_day
    )
    revisions, last_shift_revision = get_actual_revisions(
        db=session,
        global_threshold=global_threshold,
        date_from_create=date_from_create,
        schedule_id=schedule_id,
        past_revision=past_revision,
        current_revision=current_revision,
    )

    if not last_shift_revision:
        # на случай если совпадает начало расчета с началом одной из ревизий
        # тогда нам не нужно складывать в revisions, понадобится только ниже при сравнении
        last_shift_revision = list(revisions.values())[0].prev

    if last_shift.slot.interval.revision_id != last_shift_revision.id:
        # что-то где-то сломалось, нужно понять что делать
        # возможно нужно просто делать last_shift_revision = last_shift.slot.interval.revision
        logger.error(
            f'last_shift_revision: {last_shift_revision.id} not match '
            f'last_shift revision: {last_shift.slot.interval.revision_id}'
        )
        # raise InvalidData(
        #    message=(
        #        f'Revision of shift {last_shift.id}: {last_shift.slot.interval.revision_id} '
        #        f'does not match last revision:  {last_shift_revision.id}'
        #    )
        # )

    dry_shifts = create_dry_shifts(
        db=session,
        global_threshold=global_threshold,
        revisions=revisions,
        schedule=schedule,
        last_shift=last_shift,
        last_shift_revision=last_shift_revision,
        boundaries_start_date=date_from_create,
        cycle_count=cycle_count,
    )
    if dry_shifts:
        session.add_all(dry_shifts)

    if dry_shifts and any(not shift.empty for shift in dry_shifts):
        kwargs = {
            settings.FORCE_TASK_DELAY: True,
            'schedules_group_id': schedule.schedules_group_id,
            'start_date': dry_shifts[0].start,
        }
        start_people_allocation.delay(**kwargs)
    else:
        schedule.recalculation_in_process = False


@lock_task(save_metrics=True, send_to_unistat=True)
@dbconnect
def process_new_shifts_for_active_schedules(session: Session):
    for schedule in query_active_schedules_with_active_revisions(session).all():
        proceed_new_shifts.delay(schedule_id=schedule.id)


@lock_task(lock_key=lambda schedule_id, *args, **kwargs: 'delete_disabled_shifts_{}'.format(schedule_id))
@dbconnect
def delete_disabled_shifts(session: Session, schedule_id: int):
    """
    Для графика в статусе disabled:
    Удаляются все будущие смены
    Отбираем роль у текущего дежурного и переводим текущие смены в scheduled
    """
    logger.info(f'Delete and finish shifts for disabled schedule: {schedule_id}')

    date_from = now()
    schedule = get_schedule_or_404(db=session, schedule_id=schedule_id)
    if not schedule_has_shifts(session, schedule_id) or schedule.state != enums.ScheduleState.disabled:
        return

    shifts_to_delete = session.query(Shift).filter(
        Shift.schedule_id == schedule_id,
        Shift.end >= date_from
    ).options(
        joinedload(Shift.sub_shifts)
    ).all()

    to_delete = []

    # отбираем роль у дежурного в шифтах, которые еще идут, в статусе active, и где есть дежурный
    for shift in shifts_to_delete:
        if shift.status == enums.ShiftStatus.active and shift.staff_id:
            check_and_deprive_member(session=session, shift=shift)
            shift.status = enums.ShiftStatus.scheduled
        else:
            to_delete.append(shift.id)
            to_delete.extend(obj.id for obj in shift.sub_shifts)

    # удаляем будущие шифты
    session.query(Shift).filter(
        Shift.next_id.in_(to_delete)
    ).update(
        {Shift.next_id: None},
        synchronize_session=False,
    )
    session.query(Shift).filter(
        Shift.id.in_(to_delete)
    ).delete(synchronize_session=False)
