import logging

from sqlalchemy.orm import Session, joinedload

from watcher.enums import ManualGapRecurrence
from watcher.logic.manual_gap import generate_gaps
from watcher.logic.problem import resolve_problem
from watcher.logic.timezone import now
from watcher.tasks.problem import create_problems_for_staff_has_gap_shifts
from watcher.db.base import dbconnect
from watcher.db import (
    ManualGapSettings,
    ManualGap,
    Shift,
    Problem,
)

from watcher import enums

from .base import lock_task


logger = logging.getLogger(__name__)


@lock_task(lock_key=lambda gap_settings_id, *args, **kwargs: 'update_manual_gaps{}'.format(gap_settings_id))
@dbconnect
def update_manual_gaps(session: Session, gap_settings_id: int):
    logger.info('Starting updating manual gaps')
    gap_settings = (
        session.query(ManualGapSettings)
        .filter(ManualGapSettings.id == gap_settings_id)
        .options(
            joinedload(ManualGapSettings.services),
            joinedload(ManualGapSettings.schedules),
            joinedload(ManualGapSettings.gaps),
        ).first()
    )
    current_now = now()
    if gap_settings.is_active:
        to_regenerate = [obj for obj in gap_settings.gaps if obj.start >= current_now]
    else:
        to_regenerate = [obj for obj in gap_settings.gaps if obj.end >= current_now]
    to_regenerate_ids = {obj.id for obj in to_regenerate}

    # зарезолвим связанные с удаляемыми гепами проблемы в будущих шифтах
    problems_to_resolve = (
        session.query(Problem)
        .join(Shift, Shift.id == Problem.shift_id)
        .join(ManualGap, ManualGap.id == Problem.manual_gap_id)
        .filter(
            ManualGap.id.in_(to_regenerate_ids),
            Shift.status != enums.ShiftStatus.completed,
            Problem.status != enums.ProblemStatus.resolved,
        )
    )
    schedule_ids = set()
    for problem in problems_to_resolve:
        resolve_problem(problem)
        schedule_ids.add(problem.shift.schedule_id)

    # деактивируем гепы в будущем и пересоздаем их по новым настройкам
    session.query(ManualGap).filter(
        ManualGap.id.in_(to_regenerate_ids)
    ).update({
        ManualGap.is_active: False,
    }, synchronize_session=False)

    generated_gaps = generate_gaps(
        gap_settings=gap_settings,
        from_datetime=current_now,
    )
    session.bulk_insert_mappings(ManualGap, generated_gaps)
    session.commit()

    # пытаемся найти и создать проблемы по новым сгенерированным гепам
    # эта же таска вызовет перераспределение start_people_allocation, если нужно
    create_problems_for_staff_has_gap_shifts(staff_ids=[gap_settings.staff_id])
    logger.info('Finish updating manual gaps: generated %d new gaps', len(generated_gaps))


@lock_task(save_metrics=True, send_to_unistat=True)
@dbconnect
def start_generate_manual_gaps(session: Session):
    logger.info('Starting extending manual gaps')
    gaps_settings = (
        session.query(ManualGapSettings)
        .options(
            joinedload(ManualGapSettings.services),
            joinedload(ManualGapSettings.schedules),
            joinedload(ManualGapSettings.gaps),
        ).all()
    )

    current_now = now()
    to_create = list()
    to_remove = list()
    staff_ids = set()
    for obj in gaps_settings:
        if obj.staff_id not in staff_ids:
            staff_ids.add(obj.staff_id)

        last_start_date = obj.start
        for gap in obj.gaps:
            last_start_date = max(last_start_date, gap.start)
            if gap.end < current_now:
                to_remove.append(gap.id)
        if obj.recurrence is ManualGapRecurrence.once:
            continue
        new_gaps = generate_gaps(
            gap_settings=obj,
            last_start_date=last_start_date,
        )
        to_create.extend(new_gaps)

    if to_remove:
        session.query(ManualGap).filter(ManualGap.id.in_(to_remove)).update({
            ManualGap.is_active: False,
        }, synchronize_session=False)
    if to_create:
        session.bulk_insert_mappings(ManualGap, to_create)

    create_problems_for_staff_has_gap_shifts(staff_ids=staff_ids)

    logger.info('Finish extending manual gaps: generated %d new gaps', len(to_create))
