from collections import defaultdict
from dataclasses import dataclass, field
from typing import Iterable

from sqlalchemy.orm import Session, Query

from watcher import enums
from watcher.celery_app import app
from watcher.crud.composition import delete_members_from_composition
from watcher.crud.event import update_events_statuses
from watcher.crud.member import query_members_in_service_by_staff
from watcher.crud.robot import get_duty_watcher_robot
from watcher.crud.shift import query_not_completed_shifts_by_members
from watcher.db import (
    Composition,
    CompositionParticipants,
    CompositionToRole,
    CompositionToRoleExcluded,
    CompositionToStaff,
    CompositionToStaffExcluded,
    CompositionToScope,
    Shift, Member,
)
from watcher.logic.shift import bulk_set_shift_approved
from watcher.db.base import dbconnect


@dataclass
class CompositionSettings:
    roles: set[int] = field(default_factory=set)
    roles_excl: set[int] = field(default_factory=set)
    scopes: set[int] = field(default_factory=set)
    staff: set[int] = field(default_factory=set)


def _filter_members(session: Session, member_ids: Iterable[int]) -> Query:
    """
    Фильтруем участников: оставляем только тех кто кто состоит в композиции или в настройках по staff_id
    """
    queries = list()
    for table in (CompositionParticipants, CompositionToStaff, CompositionToStaffExcluded):
        queries.append(
            session.query(Member, Composition.id, Composition.full_service)
            .join(Composition, Composition.service_id == Member.service_id)
            .join(table, table.composition_id == Composition.id)
            .filter(Member.id.in_(member_ids))
            .filter(table.staff_id == Member.staff_id)
        )
    return queries[0].union(*queries[1:])


def _load_compositions_settings(
    session: Session, composition_ids: Iterable[int]
) -> defaultdict[int, CompositionSettings]:
    """
    Для каждой затронутой композиции получить информацию о настройках
    """
    settings = defaultdict(CompositionSettings)
    for table, attr, storage_attr in (
        (CompositionToRole, 'role', 'roles'),
        (CompositionToRoleExcluded, 'role', 'roles_excl'),
        (CompositionToScope, 'scope', 'scopes'),
        (CompositionToStaff, 'staff_id', 'staff')
    ):
        for obj in session.query(table).filter(table.composition_id.in_(composition_ids)):
            getattr(settings[obj.composition_id], storage_attr).add(getattr(obj, attr))
    return settings


def _group_members_compositions(
    session: Session, members_compositions: Iterable[tuple[Member, int, int]]
) -> tuple[dict[int, dict[int, set]], dict[int, dict[int, set]], dict[int, set]]:
    """
    Сгруппируем по сервису и людям, также найдем и сгруппируем композиции по серивисам
    """
    service_compositions = defaultdict(set)
    deleted_members = defaultdict(lambda: defaultdict(set))
    for member, composition_id, full_service in members_compositions:
        service_compositions[member.service_id].add((composition_id, full_service))
        deleted_members[member.service_id][member.staff_id].add(member.role)

    active_members = defaultdict(lambda: defaultdict(set))
    for service_id in service_compositions:
        staff_ids = deleted_members[service_id]
        for member in query_members_in_service_by_staff(db=session, service_id=service_id, staff_ids=staff_ids):
            active_members[member.service_id][member.staff_id].add(member.role)

    return deleted_members, active_members, service_compositions


def _select_participants_to_process(
    session: Session,
    members_compositions: Iterable[tuple[Member, int, int]],
) -> defaultdict[set[int]]:
    """
    Выбирает участников для удаления из композиций и поиска затронутых смен
    """
    deleted_members, active_members, service_compositions = _group_members_compositions(
        session=session,
        members_compositions=members_compositions
    )
    comp_settings = _load_compositions_settings(
        session=session,
        composition_ids=set([val[0] for values in service_compositions.values() for val in values])
    )
    participants_to_process = defaultdict(set)
    for service_id in service_compositions:
        for staff_id in deleted_members[service_id]:
            active_roles = active_members[service_id][staff_id]
            # Если у человека не осталось ролей в сервисе, то удаляем его из всех композиций связанных с сервисом
            if len(active_roles) == 0:
                for composition_id, _ in service_compositions[service_id]:
                    participants_to_process[composition_id].add(staff_id)
                continue
            else:
                # Можем удалить человека из CompositionParticipants в тех композициях, которые не являются
                # полными и где у него нет роли в настроках CompToRole или CompToScope (с проверкой CompToRoleExcluded)
                # Дополнительно перед этим нужно проверить что его нет в CompositionToStaff текущей композиции
                for composition_id, full_service in service_compositions[service_id]:
                    if full_service or staff_id in comp_settings[composition_id].staff:
                        continue
                    elif active_roles.intersection(comp_settings[composition_id].roles):
                        continue
                    active_scopes = {
                        role.scope for role in active_roles.difference(comp_settings[composition_id].roles_excl)
                    }
                    if active_scopes.intersection(comp_settings[composition_id].scopes):
                        continue
                    participants_to_process[composition_id].add(staff_id)

    return participants_to_process


def _delete_participants_from_compositions(session: Session, participants: defaultdict[set[int]]) -> None:
    for composition_id, staff_ids in participants.items():
        delete_members_from_composition(db=session, composition_id=composition_id, staff_ids=staff_ids)


def _find_affected_shifts(session: Session, participants: defaultdict[set[int]]) -> list[Shift]:
    shifts = list()
    for composition_id, staff_ids in participants.items():
        shifts.extend(query_not_completed_shifts_by_members(
            db=session, composition_id=composition_id, staff_ids=staff_ids
        ).all())
    return shifts


def _start_people_reallocation(session: Session, affected_shifts: list[Shift]) -> None:
    from watcher.tasks.people_allocation import start_people_allocation

    start_dates = dict()
    schedule_groups_need_reallocation = set()
    shifts_to_cancel_approve = set()
    for shift in affected_shifts:
        group_id = shift.schedule.schedules_group_id
        schedule_groups_need_reallocation.add(group_id)
        start_dates[group_id] = min(start_dates.get(group_id, shift.start), shift.start)
        if shift.approved:
            shifts_to_cancel_approve.add(shift)

    robot = get_duty_watcher_robot(session)
    if shifts_to_cancel_approve:
        bulk_set_shift_approved(db=session, db_objs=shifts_to_cancel_approve, approved=False, author_id=robot.id)
    for group_id in schedule_groups_need_reallocation:
        start_people_allocation.delay(schedules_group_id=group_id, start_date=start_dates[group_id])


@app.task
@dbconnect
def process_delete_members(session: Session, obj_to_event: dict[int: Iterable]):
    members_compositions = _filter_members(session=session, member_ids=obj_to_event.keys())
    participants_to_process = _select_participants_to_process(
        session=session,
        members_compositions=members_compositions,
    )
    _delete_participants_from_compositions(session=session, participants=participants_to_process)

    shifts = _find_affected_shifts(session=session, participants=participants_to_process)
    _start_people_reallocation(
        session=session,
        affected_shifts=shifts,
    )

    event_ids = list()
    for ev_ids in obj_to_event.values():
        event_ids.extend(ev_ids)

    update_events_statuses(db=session, ids=set(event_ids), state=enums.EventState.processed)
