from typing import List

from sqlalchemy.orm import Session
from ylog.context import log_context

from watcher import enums
from watcher.db import Shift
from watcher.db.base import dbconnect
from watcher.tasks.sync import notify_staff_duty
from watcher.crud.base import get_object_by_model
from watcher.crud.robot import get_duty_watcher_robot
from watcher.crud.schedule import query_active_schedules
from watcher.crud.shift import (
    query_need_approve_shifts_by_schedules,
    query_need_start_shifts,
    query_need_finish_shifts,
    get_simultaneous_shifts,
)
from watcher.logic.member import check_and_request_role, check_and_deprive_member
from watcher.logic.rating import update_staff_ratings_for_further_shifts
from watcher.logic.staff import find_shift_with_staff_without_intersection
from watcher.logic.shift import (
    generate_new_sequence_by_shift,
    bulk_set_shift_approved,
)
from watcher.logic.rating import update_shift_participant_rating
from .base import lock_task


import logging

logger = logging.getLogger(__name__)


@lock_task(save_metrics=True, send_to_unistat=True)
@dbconnect
def approve_schedules_shifts(session: Session, schedules_ids: List[int] = None):
    if not schedules_ids:
        schedules_ids = [schedule.id for schedule in query_active_schedules(session).all()]

    robot = get_duty_watcher_robot(session)
    need_approve_shifts = query_need_approve_shifts_by_schedules(session, schedules_ids).all()

    to_approve = set(need_approve_shifts)
    for shift in need_approve_shifts:
        to_approve.update(shift.sub_shifts)

    bulk_set_shift_approved(db=session, db_objs=to_approve, approved=True, author_id=robot.id)
    for service_id in set(shift.schedule.service_id for shift in to_approve):
        notify_staff_duty.delay(
            service_id=service_id,
        )


@lock_task(save_metrics=True, send_to_unistat=True)
@dbconnect
def start_shifts(session: Session):
    logger.info('Starting scheduling shifts')
    shifts_to_start = query_need_start_shifts(db=session).all()
    logger.info(f'Planning to start: {len(shifts_to_start)} shifts')
    for shift in shifts_to_start:
        start_shift.delay(shift_id=shift.id)

    logger.info('Finish scheduling shifts')


@lock_task(save_metrics=True, send_to_unistat=True)
@dbconnect
def finish_shifts(session: Session):
    logger.info('Starting finishing shifts')
    shifts_to_finish = query_need_finish_shifts(db=session).all()
    logger.info(f'Planning to finish: {len(shifts_to_finish)} shifts')
    for shift in shifts_to_finish:
        finish_shift.delay(shift_id=shift.id)

    logger.info('Finish finishing shifts')


@lock_task(lock_key=lambda shift_id, *args, **kwargs: 'start_shift_{}'.format(shift_id))
@dbconnect
def start_shift(session: Session, shift_id: int):
    logger.info(f'Starting shift: {shift_id}')
    shift = get_object_by_model(
        db=session, model=Shift, object_id=shift_id,
        joined_load=('slot', 'slot.interval', 'schedule', 'staff')
    )
    if not shift:
        logger.warning(f'Shift with id: {shift_id} not exist')
        return
    with log_context(shift_id=shift.id, schedule_id=shift.schedule_id):
        if shift.status != enums.ShiftStatus.scheduled:
            logger.warning(
                f'Skipping starting of not scheduled shift: {shift.id}, '
                f'actual state: {shift.status}'
            )
            return

        # назначем primary смену на текущего backup-дежурного,
        # если нет primary дежурного или он не может дежурить
        if (
            not shift.empty and shift.is_primary and
            shift.slot.interval.backup_takes_primary_shift and
            (shift.staff is None or not find_shift_with_staff_without_intersection(
                db=session, target_shift=shift, shifts=[shift]
            ))
        ):
            backup_shift = find_shift_with_staff_without_intersection(
                db=session, target_shift=shift,
                shifts=get_simultaneous_shifts(db=session, shift=shift, target_is_primary=False),
            )
            if backup_shift:
                shift.staff = backup_shift.staff
                backup_shift.gives_rating = False
                logger.info(f'Backup staff takes primary shift: primary {shift.id}, backup {backup_shift.id}')
            else:
                logger.warning(f'"Backup takes primary" is enabled but no staff available: {shift.id}')

        if not shift.staff:
            shift.status = enums.ShiftStatus.active
            if not shift.empty:
                logger.warning(f'Started not empty shift without staff: {shift.id}')
        else:
            check_and_request_role(session=session, shift=shift)


@lock_task(lock_key=lambda shift_id, *args, **kwargs: 'finish_shift_{}'.format(shift_id))
@dbconnect
def finish_shift(session: Session, shift_id: int):
    from watcher.tasks.people_allocation import start_people_allocation
    logger.info(f'Finishing shift: {shift_id}')

    shift = get_object_by_model(
        db=session, model=Shift, object_id=shift_id,
        joined_load=('slot', 'schedule', 'staff', 'next', 'sub_shifts')
    )
    if not shift:
        logger.warning(f'Shift with id: {shift_id} not exist')
        return
    with log_context(shift_id=shift.id, schedule_id=shift.schedule_id):
        if shift.status != enums.ShiftStatus.active:
            logger.warning(f'Неожиданный статус смены: {shift.status}, {shift.id}')
            return

        if not shift.staff:
            shift.status = enums.ShiftStatus.completed
            logger.info(f'Finished shift without staff: {shift.id}')
            return

        if shift.slot and shift.slot.role_on_duty_id:
            logger.info(f'Finishing shift {shift.id} with role {shift.slot.role_on_duty_id}')
            check_and_deprive_member(session=session, shift=shift)

        rating = None
        if shift.gives_rating:
            rating = update_shift_participant_rating(db=session, shift=shift)

        shift.status = enums.ShiftStatus.completed

        if not shift.next_id:
            return

        next_shift = shift.next

        if next_shift and next_shift.schedule_id != shift.schedule_id:
            logger.warning('Generating new sequence')
            next_shift = generate_new_sequence_by_shift(session, shift)

        if not next_shift:
            return

        if rating:
            rating_difference = float(
                round(
                    float(rating.rating)
                    - next_shift.predicted_ratings.get(shift.staff.login, 0),
                    1
                )
            )
            if rating_difference:
                logger.warning(
                    f'Предсказанный рейтинг сотрудника (id={shift.staff.id}) '
                    f'для смены (id={next_shift.id}) не сходится.\n'
                    f'Предсказано: {next_shift.predicted_ratings.get(shift.staff.login, 0)}\n'
                    f'Реально: {rating.rating}.\n'
                    f'Началось перераспределение людей с {shift.start}.'
                )

                inconsistent_shift = update_staff_ratings_for_further_shifts(
                    session,
                    shift,
                    {shift.staff.login: rating_difference},
                )
                if inconsistent_shift:
                    start_people_allocation.delay(
                        schedules_group_id=shift.schedule.schedules_group_id,
                        start_date=inconsistent_shift.start
                    )
