import logging
from dateutil.parser import parse

from sqlalchemy.orm import Session

from watcher import enums
from .base import lock_task
from watcher.config import settings
from watcher.db import Event, Service
from watcher.db.base import dbconnect
from watcher.logic.timezone import now
from watcher.celery_app import app
from watcher.logic.clients.staff import staff_client


logger = logging.getLogger(__name__)

DT_BATCH_SIZE = 500


@lock_task(save_metrics=True, send_to_unistat=True)
@dbconnect
def process_changes_from_logbroker(session: Session):
    from watcher.logic.logbroker import get_new_data_from_topic
    """
    В топик в логброкере
    https://logbroker.yandex-team.ru/logbroker/accounts/watcher
    пишем лог изменений данных в табличках abc
    трансфер - https://yc.yandex-team.ru/folders/fookohd8dn7lhk9k67u1/data-transfer/transfer/dttilncnhdvdtiffe50d
    эта таска их обрабатывает
    """
    try:
        data_flow = get_new_data_from_topic(
            topic=settings.LOGBROKER_ABC_TOPIC,
            consumer=settings.LOGBROKER_ABC_CONSUMER,
        )
    except Exception:
        logger.exception('Sync with logbroker failed')

    # тут рекомендуется накапливать события пачками (ну штук по 500)
    # и потом их вместе обрабатывать
    events = list()
    for table, kind, object_data, old_keys in data_flow:
        logger.info(f'Proceeding changes in table {table}, kind: {kind}')

        obj_id = object_data.get('id', object_data.get('pk', old_keys.get('id')))

        modified_at = object_data.get('modified_at', None)
        remote_modified_at = parse(modified_at) if modified_at else now()

        events.append(Event(
            table=table,
            kind=kind,
            obj_id=obj_id,
            source=enums.EventSource.logbroker,
            type=enums.EventType.db_event,
            remote_modified_at=remote_modified_at,
            object_data=object_data,
            old_keys=old_keys,
        ))

        if len(events) == DT_BATCH_SIZE:
            session.add_all(events)
            events.clear()

    if len(events):
        session.add_all(events)


@app.task
@dbconnect
def notify_staff_duty(session: Session, service_id: int) -> None:
    """Сообщаем стаффу, что ему нужно обновить у себя данные"""
    logger.info(f'Pushing staff for {service_id}')
    if not settings.NOTIFY_STAFF:
        logger.info('Skipping notify_staff, not enabled')
        return

    service = session.query(Service).get(service_id)
    if not service.is_exportable:
        logger.info(f'Skipping push for not exportable service: {service_id}')
        return

    staff_client.notify_duty(
        service_id=service_id,
        service_slug=service.slug,
    )
